/* unixfile.c -- unix file driver for options/addressbook with full locking
 *
 *	(C) Copyright 1994 by Carnegie Mellon University
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Author: Chris Newman
 * Description:
 *-------
 * An option file has syntax of the following form:
 * <option-name> <tab> <option-value>
 * Any CR, LF, or \ in the <option-value> is quoted
 * Tab, CR, LF, wildcards, and <"> are forbidden in the option-name
 *-------
 * An address book file has syntax of the following form:
 * <entry-name> <nl>
 * <tab> <field> <tab> <value>
 * Any CR, LF, or \ in the <value> is quoted
 * Tab, CR, LF, and <"> are forbidden in the entry name and field
 *-------
 * The options lock file has the following form:
 * <option-name> <tab> <user>@<hostname>
 *-------
 * The address book lock file has the following form:
 * <entry-name> <tab> <user>@<hostname>
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/param.h>
#include "mail.h"
#include "osdep.h"
#include "imap2.h"
#include "imutil.h"
#include "support.h"
#include "misc.h"
#include "glob.h"

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

/* driver name for unix option file driver */
static char drvrname[] = "UNIXFILE";

/* filenames for options */
static char Fname[] = ".imspopts";
static char Lockfname[] = ".imsplocks";
static char Abookname[] = ".abook";

/* structure to keep track of a unix file
 */
typedef struct ufile {
    unsigned long mtime;
    unsigned long size;
    long count;
    char *buf;
    keyvalue *kv;
} Ufile;

/* structure to keep track of locks held by current session
 */
typedef struct locklist {
    struct locklist *next;
    char name[1];
} locklist;

/* private abook structure
 */
typedef struct uabook {
    abook ab;
    Ufile uf, luf;
    locklist *locks;
    char *lname;
    char name[7];
} uabook;

/* unixfile option function prototypes
 */
static char *makefilename(char *);
static void readuf(char *, Ufile *);
static long lockreaduf(char *, Ufile *);
static long writeuf(char *, char *, Ufile *, int);
static long unixfile_option_get(SUPPORTSTREAM *, char *);
static long unixfile_option_lock(SUPPORTSTREAM *, char *);
static void unixfile_option_unlock(SUPPORTSTREAM *, char *);
static long unixfile_option_set(SUPPORTSTREAM *, char *, char *);
static void unixfile_option_pfree(struct optiondriver *);
static long unixfile_abook_create(SUPPORTSTREAM *, char *);
static long unixfile_abook_delete(SUPPORTSTREAM *, char *);
static long unixfile_abook_rename(SUPPORTSTREAM *, char *, char *);
static long unixfile_abook_find(SUPPORTSTREAM *, char *);
static abook *unixfile_abook_open(SUPPORTSTREAM *, char *);
static void unixfile_abook_close(abook *);
static long unixfile_abook_getlist(abook *);
static long unixfile_abook_search(abook *, char *, keyvalue *, long);
static long unixfile_abook_fetch(abook *, char *);
static long unixfile_abook_lock(abook *, char *);
static void unixfile_abook_unlock(abook *, char *);
static long unixfile_abook_store(abook *, char *, keyvalue *, long);
static long unixfile_abook_deleteent(abook *, char *);
static long unixfile_abook_getacl(abook *, int);
static long unixfile_abook_setacl(abook *, char *, char *);
static char *unixfile_abook_expand(abook *, char *);
static void unixfile_abook_pfree(struct abookdriver *);

/* private data for unixfile driver */
static Ufile user_opts = {0, 0, -1, NULL, NULL};
static Ufile user_lock = {0, 0, -1, NULL, NULL};
static locklist *olocks = NULL;
static char *home_end = NULL;
static char *user_name = NULL;
static char lock_host[MAXHOSTNAMELEN] = "";
static char fbuf[MAXPATHLEN], ftmp[MAXPATHLEN];

/* Unixfile option driver
 */
optiondriver unixfile_option_drvr = {
    drvrname,
    NULL, NULL,
    unixfile_option_get,
    unixfile_option_lock,
    unixfile_option_unlock,
    unixfile_option_set,
    unixfile_option_pfree
};

/* set the locking host
 */
void unixfile_set_lock_host(char *host)
{
    if (!host || !*host) {
	gethostname(lock_host, sizeof (lock_host));
    } else {
	strncpy(lock_host, host, sizeof (lock_host));
    }
    lock_host[sizeof (lock_host) - 1] = '\0';
}

/* make filename
 */
static char *makefilename(char *fname)
{
    struct passwd *pass;

    if (*fname != '/') {
	/* find home directory */
	if (!home_end) {
	    pass = getpwuid(geteuid());
	    if (!pass) return (fname);
	    strcpy(fbuf, pass->pw_dir);
	    strcat(fbuf, "/");
	    home_end = fbuf + strlen(fbuf);
	    user_name = cpystr(pass->pw_name);
	}
	strcpy(home_end, fname);
	fname = fbuf;
    }

    return (fname);
}

/* read an parse a unix file, if necessary
 */
static void readuf(char *fname, Ufile *uf)
{
    struct stat stbuf;
    int fd;
    char *scan, *dst;
    unsigned long count;
    keyvalue *kvpos;

    fname = makefilename(fname);
    if (uf->count >= 0 && stat(fname, &stbuf) < 0) {
	stbuf.st_mtime = uf->mtime + 1;
    }
    if ((uf->count < 0 || uf->mtime != stbuf.st_mtime)
	&& (fd = open(fname, O_RDWR|O_CREAT, 0644)) >= 0) {
	/* clear old options & read file */
	if (uf->buf) fs_give((void **) &uf->buf);
	if (uf->kv) fs_give((void **) &uf->kv);
	if (fstat(fd, &stbuf) < 0) {
	    close(fd);
	    uf->count = -1;
	    return;
	}
	uf->count = 0;
	if ((uf->size = stbuf.st_size) > 0) {
	    uf->buf = fs_get(uf->size + 1);
	    if (read(fd, uf->buf, uf->size) != uf->size) {
		fs_give((void **) &uf->buf);
		uf->count = -1;
	    }
	}
	if (uf->count >= 0) {
	    uf->mtime = stbuf.st_mtime;

	    /* parse the options */
	    uf->count = 0;
	    count = uf->size;
	    scan = uf->buf;
	    while (count--) {
		if (*scan++ == '\n') ++uf->count;
	    }
	    if (scan > uf->buf && scan[-1] != '\n') ++uf->count;
	    if (uf->count) {
		kvpos = uf->kv = fs_get(sizeof (keyvalue) * (uf->count + 1));
		count = uf->size;
		scan = uf->buf;
		do {
		    if (count && *scan == '\t') ++scan, --count;
		    kvpos->key = scan;
		    kvpos->value = NULL;
		    while (count && *scan != '\t' && *scan != '\n') {
			++scan, --count;
		    }
		    if (count && *scan == '\t') {
			*scan++ = '\0';
			kvpos->value = dst = scan;
			while (--count && *scan != '\n') {
			    /* canonicalize \n and \\ */
			    if (*scan == '\\') {
				if (!--count) break;
				if (*++scan == 'n') {
				    *dst++ = '\n';
				} else if (*scan == 'r') {
				    *dst++ = '\r';
				} else {
				    *dst++ = *scan;
				}
			    } else {
				*dst++ = *scan;
			    }
			    ++scan;
			}
			*dst = '\0';
		    }
		    ++kvpos;
		    *scan++ = '\0';
		} while (count && --count);
		uf->count = kvpos - uf->kv;
		if (!uf->count) fs_give((void **) &uf->kv);
	    }
	}
	close(fd);
    }
}

/* lock a lock file for exclusive access, and read associated file
 */
static long lockreaduf(char *fname, Ufile *uf)
{
    int lockfd, fd;

    /* get filename */
    sprintf(ftmp, "%s.lock", makefilename(fname));
    
    /* get exclusive lock */
    if ((lockfd = open(ftmp, O_RDWR|O_CREAT, 0644)) >= 0) {
	if (flock(lockfd, LOCK_EX) >= 0) {
	    readuf(fname, uf);
	    if (uf->count < 0) {
		flock(lockfd, LOCK_UN);
		close(lockfd);
		lockfd = -1;
	    }
	} else {
	    close(lockfd);
	    lockfd = -1;
	}
    } else {
	char tmp[256];
	extern int errno;

	sprintf(tmp, "failed to open address book due to unix error %d",
		errno);
	mm_log(tmp, ERROR);
    }

    return (lockfd);
}

/* write unix file
 */
static long writeuf(char *tmpfname, char *dstfname, Ufile *uf, int abook)
{
    FILE *out;
    char *newfname;
    char *start, *src, *dst;
    int i, quotecount;
    long result;

    if (tmpfname) {
	sprintf(ftmp, "%s.new", makefilename(tmpfname));
	newfname = ftmp;
    } else {
	newfname = makefilename(dstfname);
    }
    result = NIL;
    if ((out = fopen(newfname, tmpfname ? "w" : "a")) != NULL) {
	result = T;
	i = tmpfname ? -1 : uf->count - 2;
	while (++i < uf->count) {
	    if (!uf->kv[i].key) continue;
	    if (!uf->kv[i].value) {
		if (abook && fprintf(out, "%s\n", uf->kv[i].key) == EOF) {
		    result = NIL;
		}
		continue;
	    }
	    quotecount = 0;
	    start = uf->kv[i].value;
	    for (src = start; *src; ++src) {
		if (*src == '\\' || *src == '\n' || *src == '\r') {
		    ++quotecount;
		}
	    }
	    /* quote newlines and backslashes in the value */
	    if (quotecount) {
		dst = src + quotecount;
		while (src >= start) {
		    switch (*src) {
			case '\n':
			    *dst-- = 'n';  *dst-- = '\\'; break;
			case '\r':
			    *dst-- = 'r';  *dst-- = '\\'; break;
			case '\\':
			    *dst-- = '\\'; *dst-- = '\\'; break;
			default:
			    *dst-- = *src;
			    break;
		    }
		    --src;
		}
	    }
	    if (abook) putc('\t', out);
	    if (fprintf(out, "%s\t%s\n", uf->kv[i].key, start) == EOF) {
		result = NIL;
	    }
	}
	if (fclose(out) == EOF) result = NIL;
    }
    if (result == T && tmpfname
	&& rename(newfname, makefilename(dstfname)) < 0) {
	result = NIL;
    }
    if (result == NIL && tmpfname) unlink(newfname);
    if (uf->kv) fs_give((void **) &uf->kv);
    if (uf->buf) fs_give((void **) &uf->buf);
    uf->count = -1;
    
    return (result);
}

static long unixfile_option_get(SUPPORTSTREAM *s, char *pattern)
{
    glob *g;
    int i;

    readuf(Fname, &user_opts);
    if (user_opts.count < 0) {
	mm_log("Unable to access options file", ERROR);
	return (NIL);
    }
    g = glob_init(pattern, GLOB_ICASE);

    /* magic option */
    if (GLOB_TEST(g, "common.date") >= 0) {
	rfc822_date(s->imsp.reply.tmp);
	mm_option(s, "common.date", s->imsp.reply.tmp, 0);
    }

    /* user options */
    for (i = 0; i < user_opts.count; ++i) {
	if (user_opts.kv[i].value && GLOB_TEST(g, user_opts.kv[i].key) >= 0) {
	    mm_option(s, user_opts.kv[i].key, user_opts.kv[i].value, 1);
	}
    }
    glob_free(&g);

    return (T);
}

static long unixfile_option_lock(SUPPORTSTREAM *s, char *name)
{
    locklist *newlock;
    int lockfd, i;
    long result;
    char *tmpvalue;

    /* Forbid characters that don't belong in an option name
     */
    if (strpbrk(name, "\t\n\r*?%\"") != NULL) {
	sprintf(s->imsp.reply.tmp, "Illegal character in option name: %s",
		name);
	mm_log(s->imsp.reply.tmp, ERROR);
	return (NIL);
    }

    /* open and read the current locks */
    if ((lockfd = lockreaduf(Lockfname, &user_lock)) < 0) {
	mm_log("Failed to open option lock file", ERROR);
	return (NIL);
    }

    /* see if someone else holds the lock */
    for (i = 0; i < user_lock.count; ++i) {
	if (!strcasecmp(name, user_lock.kv[i].key)) break;
    }
    if (i < user_lock.count) {
	flock(lockfd, LOCK_UN);
	close(lockfd);
	sprintf(s->imsp.reply.tmp, "[LOCKED] Option `%s' already locked by %s",
		name, user_lock.kv[i].value);
	mm_log(s->imsp.reply.tmp, ERROR);
	return (NIL);
    }
    if (!user_lock.kv) user_lock.kv = (keyvalue *) fs_get(sizeof (keyvalue));

    /* add our own lock */
    if (!*lock_host) unixfile_set_lock_host(NULL);
    tmpvalue = fs_get(strlen(user_name) + strlen(lock_host) + 2);
    sprintf(tmpvalue, "%s@%s", user_name, lock_host);
    user_lock.kv[i].key = name;
    user_lock.kv[i].value = tmpvalue;
    ++user_lock.count;
    result = writeuf(NULL, Lockfname, &user_lock, 0);
    fs_give((void **) &tmpvalue);
    flock(lockfd, LOCK_UN);
    close(lockfd);

    /* add to lock list */
    if (result == T) {
	newlock = fs_get(sizeof (locklist) + strlen(name));
	strcpy(newlock->name, name);
	newlock->next = olocks;
	olocks = newlock;
    }
    
    return (result);
}

static void unixfile_option_unlock(SUPPORTSTREAM *s, char *name)
{
    locklist **plock, *curlock;
    int lockfd, i;
    long result;

    /* look for item in locklist */
    for (plock = &olocks;
	 (curlock = *plock) && strcasecmp(curlock->name, name);
	 plock = &curlock->next);
    if (!curlock) return;

    /* open and read the current locks */
    if ((lockfd = lockreaduf(Lockfname, &user_lock)) < 0) return;

    /* look for the entry */
    for (i = user_lock.count - 1; i >= 0; --i) {
	if (!strcasecmp(name, user_lock.kv[i].key)) break;
    }
    if (i < 0) {
	result = T;
    } else {
	/* remove the lock from the file */
	user_lock.kv[i].key = NULL;
	result = writeuf(Lockfname, Lockfname, &user_lock, 0);
	flock(lockfd, LOCK_UN);
	close(lockfd);
    }

    /* remove lock from list */
    if (result == T) {
	*plock = curlock->next;
	fs_give((void **) &curlock);
    }
}

static long unixfile_option_set(SUPPORTSTREAM *s, char *name, char *value)
{
    int lockfd, i;
    long result;
    char *tmpvalue, *tmpname;

    /* Forbid characters that don't belong in an option name
     */
    if (strpbrk(name, "\t\n\r?*%\"") != NULL) {
	sprintf(s->imsp.reply.tmp, "Illegal character in option name: %s",
		name);
	mm_log(s->imsp.reply.tmp, ERROR);
	return (NIL);
    }

    /* open and read the options file */
    if ((lockfd = lockreaduf(Fname, &user_opts)) < 0) {
	mm_log("Failed to lock options file for writing", ERROR);
	return (NIL);
    }

    /* see if we need to change it */
    for (i = 0; i < user_opts.count; ++i) {
	if (!strcasecmp(name, user_opts.kv[i].key)) break;
    }
    result = T;
    if ((!value && i < user_opts.count)
	|| (value && i == user_opts.count)
	|| (value && strcmp(value, user_opts.kv[i].value))) {
	if (!user_opts.kv) {
	    user_opts.kv = (keyvalue *) fs_get(sizeof (keyvalue));
	}
	/* prepare new value */
	user_opts.kv[i].key = name;
	tmpvalue = NULL;
	if (value) {
	    tmpvalue = fs_get(strlen(value) * 2 + 1);
	    strcpy(tmpvalue, value);
	} else {
	    user_opts.kv[i].key = NULL;
	}
	user_opts.kv[i].value = tmpvalue;
	tmpname = Fname;
	if (i == user_opts.count) {
	    tmpname = NULL;
	    ++user_opts.count;
	}

	/* update file */
	result = writeuf(tmpname, Fname, &user_opts, 0);
	if (tmpvalue) fs_give((void **) &tmpvalue);
    }
    flock(lockfd, LOCK_UN);
    close(lockfd);
    
    return (result);
}

/* free up options cache
 */
static void unixfile_option_pfree(struct optiondriver *odrvr)
{
    int lockfd, i;
    locklist *nextlock;
    
    /* unlock all locks held by this session */
    if (olocks) {
	if ((lockfd = lockreaduf(Lockfname, &user_lock)) >= 0) {
	    while (olocks) {
		nextlock = olocks->next;
		for (i = user_lock.count - 1; i >= 0; --i) {
		    if (!strcasecmp(olocks->name, user_lock.kv[i].key)) break;
		}
		if (i >= 0) user_lock.kv[i].key = NULL;
		fs_give((void **) &olocks);
		olocks = nextlock;
	    }
	    writeuf(Lockfname, Lockfname, &user_lock, 0);
	    flock(lockfd, LOCK_UN);
	    close(lockfd);
	}
    }

    /* free storage */
    if (user_opts.buf) fs_give((void **) &user_opts.buf);
    if (user_opts.kv) fs_give((void **) &user_opts.kv);
    user_opts.count = -1;
    home_end = NULL;
    if (user_name) fs_give((void **) &user_name);
    if (user_lock.buf) fs_give((void **) &user_lock.buf);
    if (user_lock.kv) fs_give((void **) &user_lock.kv);
    user_lock.count = -1;
}

/*************************
 * Address book routines *
 *************************/

/* Unixfile address book driver
 */
abookdriver unixfile_abook_drvr = {
    drvrname,
    NULL, NULL,
    unixfile_abook_create,
    unixfile_abook_delete,
    unixfile_abook_rename,
    unixfile_abook_find,
    unixfile_abook_open,
    unixfile_abook_close,
    unixfile_abook_getlist,
    unixfile_abook_search,
    unixfile_abook_fetch,
    unixfile_abook_lock,
    unixfile_abook_unlock,
    unixfile_abook_store,
    unixfile_abook_deleteent,
    unixfile_abook_expand,
    unixfile_abook_getacl,
    unixfile_abook_setacl,
    unixfile_abook_pfree
};

/* read an abook into internal buffer
 *  report an error on failure.
 */
static long readabook(uabook *uab)
{
    readuf(uab->name, &uab->uf);
    if (uab->uf.count < 0) {
	sprintf(uab->ab.parent->imsp.reply.tmp,
		"Failed to open address book `%s'", uab->ab.name);
	mm_log(uab->ab.parent->imsp.reply.tmp, ERROR);
	return (NIL);
    }
    
    return (T);
}

/* abook driver functions:
 */
static long unixfile_abook_create(SUPPORTSTREAM *s, char *name)
{
    mm_log("Multiple address books not currently supported", ERROR);
    return (NIL);
}
static long unixfile_abook_delete(SUPPORTSTREAM *s, char *name)
{
    mm_log("Multiple address books not currently supported", ERROR);
    return (NIL);
}
static long unixfile_abook_rename(SUPPORTSTREAM *s,
				  char *oldname, char *newname)
{
    mm_log("Multiple address books not currently supported", ERROR);
    return (NIL);
}
static long unixfile_abook_find(SUPPORTSTREAM *s, char *pattern)
{
    glob *g;

    if (!user_name) (void) makefilename(Abookname);

    g = glob_init(pattern, GLOB_ICASE);
    if (GLOB_TEST(g, user_name) >= 0) mm_addressbook(s, user_name, '.');
    glob_free(&g);
    
    return (T);
}
static abook *unixfile_abook_open(SUPPORTSTREAM *s, char *name)
{
    char *fname = makefilename(Abookname);
    uabook *uab;

    /* validate name:
     *  Note that currently the only valid name is the default address book
     *  with the user's user_name as it's name.  This may be changed at a
     *  later date, and the following code is written with that intention.
     */
    if (!name) {
	name = user_name;
    } else if (strcasecmp(name, user_name)) {
	sprintf(s->imsp.reply.tmp,
		"`%s' is an invalid address book name", name);
	mm_log(s->imsp.reply.tmp, ERROR);

	return (NULL);
    }

    /* create address book structure */
    uab = (uabook *) fs_get(sizeof (uabook) + strlen(fname) * 2);
    uab->ab.name = user_name;
    strcpy(uab->name, fname);
    uab->lname = uab->name + strlen(fname) + 1;
    sprintf(uab->lname, "%slocks", fname);
    uab->uf.buf = NULL;
    uab->uf.kv = NULL;
    uab->uf.count = -1;
    uab->luf.buf = NULL;
    uab->luf.kv = NULL;
    uab->luf.count = -1;
    uab->locks = NULL;
    
    return ((abook *) uab);
}
static void unixfile_abook_close(abook *ab)
{
    uabook *uab = (uabook *) ab;
    int lockfd, i;
    locklist *nextlock;

    /* unlock any session locks */
    if (uab->locks) {
	if ((lockfd = lockreaduf(uab->lname, &uab->luf)) >= 0) {
	    while (uab->locks) {
		nextlock = uab->locks->next;
		for (i = uab->luf.count - 1; i >= 0; --i) {
		    if (!strcasecmp(uab->locks->name, uab->luf.kv[i].key))
			break;
		}
		if (i >= 0) uab->luf.kv[i].key = NULL;
		fs_give((void **) &uab->locks);
		uab->locks = nextlock;
	    }
	    writeuf(uab->lname, uab->lname, &uab->luf, 0);
	    flock(lockfd, LOCK_UN);
	    close(lockfd);
	}
    }

    /* free storage */
    if (uab->luf.buf) fs_give((void **) uab->luf.buf);
    if (uab->luf.kv) fs_give((void **) &uab->luf.kv);
    if (uab->uf.buf) fs_give((void **) &uab->uf.buf);
    if (uab->uf.kv) fs_give((void **) &uab->uf.kv);
    fs_give((void **) &uab);
}
static long unixfile_abook_getlist(abook *ab)
{
    uabook *uab = (uabook *) ab;
    int i;
    
    if (!readabook(uab)) return (NIL);
    for (i = 0; i < uab->uf.count; ++i) {
	if (!uab->uf.kv[i].value) {
	    mm_address(ab, uab->uf.kv[i].key, NULL, 0);
	}
    }

    return (T);
}
static long unixfile_abook_search(abook *ab, char *name,
				  keyvalue *kv, long count)
{
    uabook *uab = (uabook *) ab;
    keyvalue *ukv;
    glob *g, *vg;
    int i, j, k;
    
    if (!readabook(uab)) return (NIL);
    ukv = uab->uf.kv;
    g = glob_init(name, GLOB_ICASE);
    for (i = 0; i < uab->uf.count; ++i) {
	if (!ukv[i].value && GLOB_TEST(g, ukv[i].key) >= 0) {
	    for (j = 0; j < count; ++j) {
		vg = glob_init(kv[j].value);
		for (k = i + 1; k < uab->uf.count && ukv[k].value; ++k) {
		    if (!strcasecmp(ukv[k].key, kv[j].key)
			&& GLOB_TEST(vg, ukv[k].value) >= 0) {
			break;
		    }
		}
		glob_free(&vg);
		if (k == uab->uf.count || !ukv[k].value) break;
	    }
	    if (j == count) mm_address(ab, ukv[i].key, NULL, 0);
	}
    }
    glob_free(&g);

    return (T);
}
static long unixfile_abook_fetch(abook *ab, char *name)
{
    uabook *uab = (uabook *) ab;
    int i, j, count;

    if (!readabook(uab)) return (NIL);
    for (i = 0; i < uab->uf.count; ++i) {
	if (!uab->uf.kv[i].value && !strcasecmp(name, uab->uf.kv[i].key)) {
	    for (j = i + 1; j < uab->uf.count && uab->uf.kv[j].value; ++j);
	    if ((count = j - i - 1) > 0) {
		mm_address(ab, uab->uf.kv[i].key, uab->uf.kv + i + 1, count);
	    }
	    break;
	}
    }

    return (T);
}
static long unixfile_abook_lock(abook *ab, char *name)
{
    uabook *uab = (uabook *) ab;
    locklist *newlock;
    char *tmpvalue;
    int lockfd, i;
    long result;
    
    /* verify this is a valid change */
    if (strpbrk(name, "\t\n\r\"") != NULL) {
	sprintf(ab->parent->imsp.reply.tmp,
		"Illegal character in address book entry name: %s", name);
	mm_log(ab->parent->imsp.reply.tmp, ERROR);
	return (NIL);
    }

    /* lock & read advisory locks */
    if ((lockfd = lockreaduf(uab->lname, &uab->luf)) < 0) {
	sprintf(ab->parent->imsp.reply.tmp,
		"Failed to lock address book `%s' for changes", ab->name);
	mm_log(ab->parent->imsp.reply.tmp, ERROR);
	return (NIL);
    }

    /* see if someone else holds the lock */
    for (i = 0; i < uab->luf.count; ++i) {
	if (!strcasecmp(name, uab->luf.kv[i].key)) break;
    }
    if (i < uab->luf.count) {
	flock(lockfd, LOCK_UN);
	close(lockfd);
	sprintf(ab->parent->imsp.reply.tmp, "[LOCKED] Entry `%s' in address book `%s' already locked by %s", name, ab->name, uab->luf.kv[i].value);
	mm_log(ab->parent->imsp.reply.tmp, ERROR);
	return (NIL);
    }
    if (!uab->luf.kv) uab->luf.kv = (keyvalue *) fs_get(sizeof (keyvalue));
    
    /* add our own lock */
    if (!*lock_host) unixfile_set_lock_host(NULL);
    tmpvalue = fs_get(strlen(user_name) + strlen(lock_host) + 2);
    sprintf(tmpvalue, "%s@%s", user_name, lock_host);
    uab->luf.kv[i].key = name;
    uab->luf.kv[i].value = tmpvalue;
    ++uab->luf.count;
    result = writeuf(NULL, uab->lname, &uab->luf, 0);
    fs_give((void **) &tmpvalue);
    flock(lockfd, LOCK_UN);
    close(lockfd);

    /* add to lock list */
    if (result == T) {
	newlock = fs_get(sizeof (locklist) + strlen(name));
	strcpy(newlock->name, name);
	newlock->next = uab->locks;
	uab->locks = newlock;
    }

    return (T);
}
static void unixfile_abook_unlock(abook *ab, char *name)
{
    uabook *uab = (uabook *) ab;
    locklist **plock, *curlock;
    int lockfd, i;
    long result;

    /* look for item in locklist */
    for (plock = &uab->locks;
	 (curlock = *plock) && strcasecmp(curlock->name, name);
	 plock = &curlock->next);
    if (!curlock) return;

    /* open and read the current locks */
    if ((lockfd = lockreaduf(uab->lname, &uab->luf)) < 0) return;

    /* look for the entry */
    for (i = uab->luf.count - 1; i >= 0; --i) {
	if (!strcasecmp(name, uab->luf.kv[i].key)) break;
    }
    if (i < 0) {
	result = T;
    } else {
	/* remove the lock from the file */
	uab->luf.kv[i].key = NULL;
	result = writeuf(uab->lname, uab->lname, &uab->luf, 0);
	flock(lockfd, LOCK_UN);
	close(lockfd);
    }

    /* remove lock from list */
    if (result == T) {
	*plock = curlock->next;
	fs_give((void **) &curlock);
    }
}
static long unixfile_abook_store(abook *ab, char *name,
				 keyvalue *kv, long count)
{
    uabook *uab = (uabook *) ab;
    keyvalue *newent, *ukv;
    char **strlist, **newstr;
    int lockfd, i, j, k;
    long result = NIL;

    /* verify this is a valid change */
    if (strpbrk(name, "\t\n\r\"") != NULL) {
	sprintf(ab->parent->imsp.reply.tmp,
		"Illegal character in address book entry name: %s", name);
	mm_log(ab->parent->imsp.reply.tmp, ERROR);
	return (NIL);
    }
    
    /* lock & read address book */
    if ((lockfd = lockreaduf(uab->name, &uab->uf)) < 0) {
	sprintf(ab->parent->imsp.reply.tmp,
		"Failed to lock address book `%s' for changes", ab->name);
	mm_log(ab->parent->imsp.reply.tmp, ERROR);
	return (NIL);
    }

    /* deal with an empty address book */
    if (uab->uf.kv == NULL) {
	uab->uf.kv = (keyvalue *) fs_get(sizeof (keyvalue));
    }
    ukv = uab->uf.kv;

    /* find the entry */
    for (i = 0; i < uab->uf.count; ++i) {
	if (!ukv[i].value && !strcasecmp(name, ukv[i].key)) {
	    break;
	}
    }

    /* if we're not adding new data, it's an error if there's no old entry */
    for (j = 0; j < count && !*kv[j].value; ++j);
    if (j == count && i == uab->uf.count) {
	sprintf(ab->parent->imsp.reply.tmp,
		"`%s' not found in address book `%s'", name, ab->name);
	mm_log(ab->parent->imsp.reply.tmp, ERROR);
	flock(lockfd, LOCK_UN);
	close(lockfd);
	return (NIL);
    }

    /* create the new entry */
    strlist = NULL;
    if (count) {
	/* find the end of the old entry */
	for (j = i + 1; j < uab->uf.count && ukv[j].value; ++j);

	/* make space for a new entry */
	fs_resize((void **) &uab->uf.kv, sizeof (keyvalue)
		  * (uab->uf.count + count + j - i));
	newstr = strlist = (char **) fs_get(sizeof (char *) * (count + j - i));
	ukv = uab->uf.kv;
	newent = ukv + uab->uf.count;
	newent->key = i < uab->uf.count ? ukv[i].key : name;
	newent->value = NULL;
	++newent;

	/* copy new values to new entry */
	while (count--) {
	    if (strpbrk(kv->key, "\t\n\r") != NULL) {
		sprintf(ab->parent->imsp.reply.tmp,
			"Illegal address book field `%s' ignored", kv->key);
		mm_log(ab->parent->imsp.reply.tmp, WARN);
		++kv;
		continue;
	    }
	    *newstr = fs_get(strlen(kv->value) * 2 + 1);
	    newent->key = kv->key;
	    newent->value = strcpy(*newstr++, kv->value);
	    ++kv;
	    for (k = i + 1; k < j; ++k) {
		if (ukv[k].key && !strcasecmp(newent->key, ukv[k].key)) {
		    ukv[k].key = NULL;
		}
	    }
	    if (*newent->value) ++newent;
	}

	/* copy old values to new entry */
	for (k = i + 1; k < j; ++k) {
	    if (ukv[k].key && *ukv[k].value) {
		*newstr = fs_get(strlen(ukv[k].value) * 2 + 1);
		newent->key = ukv[k].key;
		newent->value = strcpy(*newstr++, ukv[k].value);
		++newent;
	    }
	}
    }

    /* remove the old entry */
    if (i < uab->uf.count) {
	do {
	    ukv[i].key = NULL;
	} while (++i < uab->uf.count && ukv[i].value);
    }

    /* update size */
    if (count) uab->uf.count = newent - ukv;
    
    /* update database */
    result = writeuf(uab->name, uab->name, &uab->uf, 1);

    /* cleanup */
    if (strlist) {
	for (i = 0; strlist + i < newstr; ++i) {
	    fs_give((void **) (strlist + i));
	}
	fs_give((void **) &strlist);
    }
    flock(lockfd, LOCK_UN);
    close(lockfd);

    return (result);
}
static long unixfile_abook_deleteent(abook *ab, char *name)
{
    return (unixfile_abook_store(ab, name, NULL, 0));
}
static char *unixfile_abook_expand(abook *ab, char *name)
{
    uabook *uab = (uabook *) ab;
    int i, j;
    char *result = NULL;

    if (!readabook(uab)) return (result);
    for (i = 0; i < uab->uf.count; ++i) {
	if (!uab->uf.kv[i].value && !strcasecmp(name, uab->uf.kv[i].key)) {
	    for (j = i + 1; j < uab->uf.count && uab->uf.kv[j].value; ++j) {
		/*XXX: need to deal with "members" field */
		if (!strcasecmp("email", uab->uf.kv[j].key)) {
		    result = uab->uf.kv[j].value;
		    break;
		}
	    }
	    break;
	}
    }

    return (result);
}
static long unixfile_abook_getacl(abook *ab, int myacl)
{
    /* Not yet supported */
    return (NIL);
}
static long unixfile_abook_setacl(abook *ab, char *ident, char *rights)
{
    /* Not yet supported */
    return (NIL);
}
static void unixfile_abook_pfree(struct abookdriver *adrvr)
{
    home_end = NULL;
    if (user_name) fs_give((void **) &user_name);
}
