/* support.c -- Support stream routines for c-client
 *
 *	(C) Copyright 1993-1994 by Carnegie Mellon University
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Author: Chris Newman
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "mail.h"
#include "osdep.h"
#include "imap2.h"
#include "imutil.h"
#include "support.h"

/* create a new SUPPORTSTREAM
 */
SUPPORTSTREAM *support_new()
{
    SUPPORTSTREAM *s;

    s = (SUPPORTSTREAM *) fs_get(sizeof (SUPPORTSTREAM));
    s->next = (SUPPORTSTREAM *) 0;
    s->adrvr = (abookdriver *) 0;
    s->odrvr = (optiondriver *) 0;
    s->ab = (abook *) 0;
    s->mstream = (MAILSTREAM *) 0;
    s->imsp.tcp = (void *) 0;
    s->imsp.reply.line = (void *) 0;
    s->imsp.reply.extra = (void *) 0;
    s->imsp.gensym = 1;
    s->imsp.parentstruct = s;
    s->imsp.username = (void *) 0;
    s->imsp.wksize = 0;

    return (s);
}

/* destroy a SUPPORTSTREAM
 */
void support_done(SUPPORTSTREAM *s)
{
    abook *ab, *nextab;
    abookdriver *advr, *nextadvr;
    optiondriver *odvr, *nextodvr;
    MAILSTREAM *mstream, *nextmstream;
    IMEXTRALINE *extra;
    
    while (ab = s->ab) {
	nextab = ab->next;
	(*ab->adrvr->close)(ab);
	s->ab = nextab;
    }
    while (advr = s->adrvr) {
	nextadvr = advr->next;
	(*advr->pfree)(advr);
	s->adrvr = nextadvr;
    }
    while (odvr = s->odrvr) {
	nextodvr = odvr->next;
	(*odvr->pfree)(odvr);
	s->odrvr = nextodvr;
    }
    while (mstream = s->mstream) {
	nextmstream = ((SUPPORTLOCAL *) mstream->local)->next;
	(*mstream->dtb->close)(mstream);
	s->mstream = nextmstream;
    }
    if (s->imsp.reply.line) {
	fs_give((void **) &s->imsp.reply.line);
    }
    if (s->imsp.reply.extra) {
	while (extra = s->imsp.reply.extra) {
	    s->imsp.reply.extra = extra->next;
	    if (extra->line) fs_give((void **) &extra->line);
	    fs_give((void **) &extra);
	}
    }
    if (s->imsp.username) {
	fs_give((void **) &s->imsp.username);
    }
    if (s->imsp.wksize > 0) {
	fs_give((void **) &s->imsp.wkspace);
    }
    if (s->imsp.tcp) {
	tcp_close(s->imsp.tcp);
    }
    fs_give((void **) &s);
}

/* add an address book driver
 */
void support_add_adriver(SUPPORTSTREAM *s, abookdriver *adrvr)
{
    abookdriver **pdrvr;

    if (adrvr) {
	for (pdrvr = &s->adrvr; *pdrvr; pdrvr = &(*pdrvr)->next);
	*pdrvr = adrvr;
	adrvr->next = NULL;
    }
}

/* add an options driver
 */
void support_add_odriver(SUPPORTSTREAM *s, optiondriver *odrvr)
{
    optiondriver **pdrvr;

    if (odrvr) {
	for (pdrvr = &s->odrvr; *pdrvr; pdrvr = &(*pdrvr)->next);
	*pdrvr = odrvr;
	odrvr->next = NULL;
    }
}

/**************************
 * option driver routines *
 **************************/

/* get an option
 *  returns: T on success, NIL on failure
 */
long option_get(SUPPORTSTREAM *s, char *pattern)
{
    if (s && pattern && s->odrvr && s->odrvr->get) {
	s->current_odrvr = s->odrvr;
	return ((*s->odrvr->get)(s, pattern));
    }

    return (NIL);
}

/* lock an option
 *  returns: T on success, NIL on failure
 */
long option_lock(SUPPORTSTREAM *s, char *option)
{
    if (s && option && s->odrvr && s->odrvr->lock) {
	s->current_odrvr = s->odrvr;
	return ((*s->odrvr->lock)(s, option));
    }

    return (NIL);
}

/* unlock an option
 */
void option_unlock(SUPPORTSTREAM *s, char *option)
{
    if (s && option && s->odrvr && s->odrvr->unlock) {
	s->current_odrvr = s->odrvr;
	(*s->odrvr->unlock)(s, option);
    }
}

/* set an option
 *  returns: T on success, NIL on failure
 */
long option_set(SUPPORTSTREAM *s, char *option, char *value)
{
    if (s && option && s->odrvr && s->odrvr->set) {
	s->current_odrvr = s->odrvr;
	return ((*s->odrvr->set)(s, option, value));
    }

    return (NIL);
}

/*******************************
 * addressbook driver routines *
 *******************************/

/* create address book
 *  returns: T on success, NIL on failure
 */
long abook_create(SUPPORTSTREAM *s, char *name)
{
    abookdriver *adrvr;
    int status = NIL;

    if (s) {
	for (s->current_adrvr = adrvr = s->adrvr; adrvr &&
	     (!adrvr->create || (status = (*adrvr->create)(s, name)) == NIL);
	     s->current_adrvr = adrvr = adrvr->next);
    }

    return (status);
}

/* delete address book
 *  returns: T on success, NIL on failure
 */
long abook_delete(SUPPORTSTREAM *s, char *name)
{
    abookdriver *adrvr;
    int status = NIL;

    if (s) {
	for (s->current_adrvr = adrvr = s->adrvr; adrvr &&
	     (!adrvr->delete || (status = (*adrvr->delete)(s, name)) == NIL);
	     s->current_adrvr = adrvr = adrvr->next);
    }

    return (status);
}

/* rename address book
 *  returns: T on success, NIL on failure
 */
long abook_rename(SUPPORTSTREAM *s, char *oldname, char *newname)
{
    abookdriver *adrvr;
    int status = NIL;

    if (s) {
	for (s->current_adrvr = adrvr = s->adrvr; adrvr &&
	     (!adrvr->rename
	      || (status = (*adrvr->rename)(s, oldname, newname)) == NIL);
	     s->current_adrvr = adrvr = adrvr->next);
    }

    return (status);
}

/* find available address books
 *  returns: T on success, NIL on failure
 */
long abook_find(SUPPORTSTREAM *s, char *pattern)
{
    abookdriver *adrvr;
    int status = NIL;

    if (s) {
	for (s->current_adrvr = adrvr = s->adrvr;
	     adrvr && (!adrvr->find || (status = (*adrvr->find)(s, pattern)));
	     s->current_adrvr = adrvr = adrvr->next);
    }

    return (status);
}

/* open an address book
 */
abook *abook_open(SUPPORTSTREAM *s, char *abook_name, void *generic)
{
    abookdriver *adrvr;
    abook *ab = NULL;

    if (s) {
	for (s->current_adrvr = adrvr = s->adrvr;
	     adrvr && (!adrvr->open || !(ab = (*adrvr->open)(s, abook_name)));
	     s->current_adrvr = adrvr = adrvr->next);
	if (ab) {
	    ab->parent = s;
	    ab->adrvr = adrvr;
	    ab->generic = generic;
	    ab->next = s->ab;
	    s->ab = ab;
	}
    }

    return (ab);
}

/* close an address book
 */
void abook_close(abook *ab)
{
    abook **pab;
    
    for (pab = &ab->parent->ab; *pab && *pab != ab; pab = &(*pab)->next);
    if (*pab) *pab = ab->next;
    if (ab && ab->adrvr && ab->adrvr->close) (*ab->adrvr->close)(ab);
}

/* get the list of address book entries
 *  returns: T on success, NIL on failure
 */
long abook_getlist(abook *ab)
{
    if (ab && ab->adrvr && ab->adrvr->getlist) {
	return ((*ab->adrvr->getlist)(ab));
    }

    return (NIL);
}

/* search the address book
 *  returns: T on success, NIL on failure
 */
long abook_search(abook *ab, char *pattern, keyvalue *criteria, long num)
{
    if (ab && ab->adrvr && ab->adrvr->search
	&& pattern && (!num || criteria)) {
	return ((*ab->adrvr->search)(ab, pattern, criteria, num));
    }

    return (NIL);
}

/* get an address book entry
 *  returns: T on success, NIL on failure
 */
long abook_fetch(abook *ab, char *name)
{
    if (ab && ab->adrvr && ab->adrvr->fetch) {
	return ((*ab->adrvr->fetch)(ab, name));
    }

    return (NIL);
}

/* lock an address book entry
 *  returns: T on success, NIL on failure
 */
long abook_lock(abook *ab, char *name)
{
    if (ab && ab->adrvr && ab->adrvr->lock) {
	return ((*ab->adrvr->lock)(ab, name));
    }

    return (NIL);
}

/* unlock an address book entry
 */
void abook_unlock(abook *ab, char *name)
{
    if (ab && ab->adrvr && ab->adrvr->unlock) (*ab->adrvr->unlock)(ab, name);
}

/* store an address book entry
 *  returns: T on success, NIL on failure
 */
long abook_store(abook *ab, char *name, keyvalue *fielddata, long num)
{
    if (ab && ab->adrvr && ab->adrvr->store && name && fielddata && num > 0) {
	return ((*ab->adrvr->store)(ab, name, fielddata, num));
    }

    return (NIL);
}

/* delete an address book entry
 *  returns: T on success, NIL on failure
 */
long abook_deleteent(abook *ab, char *name)
{
    if (ab && ab->adrvr && ab->adrvr->deleteent) {
	return ((*ab->adrvr->deleteent)(ab, name));
    }

    return (NIL);
}

/* expand an address book entry
 *  returns: expanded address on success, NULL on failure
 */
char *abook_expand(abook *ab, char *name)
{
    if (ab && ab->adrvr && ab->adrvr->expand) {
	return ((*ab->adrvr->expand)(ab, name));
    }

    return (NULL);
}

/* get an access control list
 *  returns: T on success, NIL on failure
 */
long abook_getacl(abook *ab, int myrights)
{
    if (ab && ab->adrvr && ab->adrvr->getacl) {
	return ((*ab->adrvr->getacl)(ab, myrights));
    }

    return (NULL);
}

/* get an access control list
 *  returns: T on success, NIL on failure
 */
long abook_setacl(abook *ab, char *identifier, char *rights)
{
    if (ab && ab->adrvr && ab->adrvr->setacl) {
	return ((*ab->adrvr->setacl)(ab, identifier, rights));
    }

    return (NULL);
}
