/* imutil.c -- IMSP utility functions
 *
 *	(C) Copyright 1993-1994 by Carnegie Mellon University
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Author: Chris Newman
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include "mail.h"
#include "osdep.h"
#include "imap2.h"
#include "imutil.h"
#include "support.h"
#include "imsp.h"

/* keyword list
 */
static struct im_keyword {
    char *keyword;
    long code;
} im_keylist[] = {
    "OK", IMKEY_OK,
    "NO", IMKEY_NO,
    "ADDRESSBOOK", IMKEY_ADDRESSBOOK,
    "SEARCHADDRESS", IMKEY_SEARCHADDRESS,
    "FETCHADDRESS", IMKEY_FETCHADDRESS,
    "OPTION", IMKEY_OPTION,
    "MYRIGHTS", IMKEY_MYRIGHTS,
    "ACL", IMKEY_ACL,
    "FETCH", IMKEY_FETCH,
    "SEARCH", IMKEY_SEARCH,
    "MAILBOX", IMKEY_MAILBOX,
    "BBOARD", IMKEY_BBOARD,
    "EXISTS", IMKEY_EXISTS,
    "FLAGS", IMKEY_FLAGS,
    "RECENT", IMKEY_RECENT,
    "BAD", IMKEY_BAD,
    "EXPUNGE", IMKEY_EXPUNGE,
    "BYE", IMKEY_BYE,
    "PREAUTH", IMKEY_PREAUTH,
    "COPY", IMKEY_COPY,
    "STORE", IMKEY_STORE,
    NULL, IMKEY_UNKNOWN
};

/* bit 1: valid atom CHARACTER
 * bit 2: valid quoted string CHARACTER
 */
char im_table[256] = {
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    2, 3, 0, 3, 3, 2, 3, 3, 3, 3, 2, 3, 3, 3, 3, 3,
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 0, 3, 3, 3,
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 2, 3, 3, 3, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
};

/* used by im_keyvalue */
#define STARTING_KEYVALUE_COUNT 64

/* used by im_send */
#define MAX_INTEGER_LEN   24	/* max length of an integer */
#define MAX_LITERAL_EXTRA 32	/* max extra space needed for a literal */

/* parse an IMAP/IMSP reply
 */
IMPARSEDREPLY *im_parse_reply(char *text, IMPARSEDREPLY *reply)
{
    char *closebrace;
    struct im_keyword *kword = im_keylist;
    IMEXTRALINE *extra;

    /* initialize and free parsed reply structure */
    if (reply && reply->line) {
	fs_give((void **) &reply->line);
	while (reply->extra) {
	    extra = reply->extra;
	    reply->extra = extra->next;
	    if (extra->line) fs_give((void **) &extra->line);
	    fs_give((void **) &extra);
	}
    }
    if (!(reply->line = text)) return (NULL);
    reply->key = NULL;
    reply->text = NULL;
    reply->arg = -1;
    reply->keyval = IMKEY_UNKNOWN;

    /* get tag & keyword -- make sure they're valid */
    if (!(reply->tag = (char *) strtok(reply->line, " "))) {
	sprintf(reply->tmp, "server sent a blank line");
	mm_log(reply->tmp, WARN);
	return (NULL);
    }
    if (*reply->tag == '+') {
	reply->key = "";
	reply->text = (char *) strtok(NULL, "\n");
	return (reply);
    }
    if (!(reply->key = (char *) strtok(NULL, " "))) {
	sprintf(reply->tmp, "Missing server keyword: %.80s", reply->tag);
	mm_log(reply->tmp, WARN);
	return (NULL);
    }

    /* check for numeric arguments */
    if (isdigit(*reply->key)) {
	reply->arg = atoi(reply->key);
	if (!(reply->key = (char *) strtok(NULL, " "))) {
	    sprintf(reply->tmp,
		    "Missing keyword after numeric argument: %.80s",
		    reply->arg);
	    mm_log(reply->tmp, WARN);
	    return (NULL);
	}
    }
    ucase(reply->key);

    /* allow empty text */
    if (!(reply->text = (char *) strtok(NULL, "\n"))) {
	reply->text = reply->key + strlen(reply->key);
    }

    /* look up the keyword */
    while (kword->keyword != NULL && strcmp(kword->keyword, reply->key))
	++kword;
    reply->keyval = kword->code;
    
    /* do error handling */
    if (!reply->keyval) {
	sprintf(reply->tmp, "Unknown keyword: %.80s", reply->key);
	mm_log(reply->tmp, WARN);
	return (NULL);
    }
    if (reply->keyval >= IMKEY_EXISTS && reply->keyval <= IMKEY_STORE
	&& reply->arg < 0) {
	sprintf(reply->tmp, "keyword '%s' requries numeric argument",
		reply->key);
	mm_log(reply->tmp, WARN);
	return (NULL);
    }

    return (reply);
}

/* Internet Message Protocols: get a string
 */
char *im_string(IMSTREAMDATA *data)
{
    char *src = data->reply.text;
    char *result, *dst;
    IMEXTRALINE *extra;
    long len;

    if (!src || !*src) return (NULL);
    if (*src == '{') {
	/* process a literal */
	len = atol(src + 1);
	if (len) {
	    /*XXX: consider using Mark's "STRING" package for small machines */
	    extra = fs_get(sizeof (IMEXTRALINE) + len);
	    extra->next = data->reply.extra;
	    data->reply.extra = extra;
	    result = extra->literal;
	    if (tcp_getbuffer(data->tcp, len, result)) {
		result[len] = '\0';
	    }
	    src = extra->line = tcp_getline(data->tcp);
	    if (*src == ' ') ++src;
	}
    } else if (*src == '"') {
	/* process a quoted string */
	result = ++src;
	while (*src && *src != '"' && *src != '\\') ++src;
	if (*src == '\\') {
	    for (dst = src; *src && *src != '"'; *dst++ = *src++)
		if (*src == '\\') ++src;
	    *dst = '\0';
	}
	if (*src) *src++ = '\0';
	if (*src == ' ') ++src;
    } else {
	/* process an atom */
	result = src;
	while (*src && *src != ' ') ++src;
	if (*src) *src++ = '\0';
    }
    data->reply.text = src;

    return (result);
}

/* Internet Message Protocols: read a key-value structure
 */
keyvalue *im_keyvalue(IMSTREAMDATA *data, long *kvcount)
{
    keyvalue *kv;
    long max, pos;

    max = STARTING_KEYVALUE_COUNT;
    pos = 0;
    kv = fs_get(sizeof (keyvalue) * max);
    while (kv[pos].key = im_string(data)) {
	kv[pos].value = im_string(data);
	if (++pos == max) {
	    fs_resize((void **) &kv, sizeof (keyvalue) * (max *= 2));
	}
    }
    if (!(*kvcount = pos)) fs_give((void **) &kv);

    return (kv);
}

/* Internet Message Protocols: get reply
 * Accepts: pointer to stream data
 *	    tag to search or NIL if want a greeting
 * Returns: T if operation succeeded (OK/PREAUTH/+), NIL if it failed
 */
long im_reply(IMSTREAMDATA *data, char *tag)
{
    IMPARSEDREPLY *reply = &data->reply;
    char *str1, *str2, *str3, *str4;
    keyvalue *kv;
    long kvcount;
    extern void imsp_fetchaddress();
    
    reply->text = "Connection broken in reply";
    while (data->tcp) {
	if (!im_parse_reply(tcp_getline(data->tcp), reply)) {
	    if (!reply->line) {
		tcp_close(data->tcp);
		data->tcp = NULL;
	    }
	} else if (!strcmp(reply->tag, "*")) {
	    if (!tag) {
		return (reply->keyval == IMKEY_OK
			|| reply->keyval == IMKEY_PREAUTH);
	    }
	    switch (reply->keyval) {
		case IMKEY_OK:
		    mm_log(reply->text, NIL);
		    break;
		case IMKEY_NO:
		    mm_log(reply->text, WARN);
		    break;
		case IMKEY_BAD:
		    mm_log(reply->text, ERROR);
		    break;
		case IMKEY_ADDRESSBOOK:
		    /* XXX: need to do the list-skipping properly: */
		    while ((str1 = im_string(data))
			   && str1[strlen(str1) - 1] != ')');
		    str2 = im_string(data);
		    str3 = im_string(data);
		    if (str3) {
			mm_addressbook((SUPPORTSTREAM *) data->parentstruct,
				       str3, str2 ? *str2 : '\0');
		    }
		    break;
		case IMKEY_SEARCHADDRESS:
		    mm_address(((SUPPORTSTREAM *)data->parentstruct)->activeab,
			       im_string(data), NULL, 0);
		    break;
		case IMKEY_FETCHADDRESS:
		    str1 = im_string(data);
		    str2 = im_string(data);
		    kv = im_keyvalue(data, &kvcount);
		    imsp_fetchaddress((SUPPORTSTREAM *) data->parentstruct,
				      str1, str2, kv, kvcount);
		    if (kv) fs_give((void **) &kv);
		    break;
		case IMKEY_OPTION:
		    str1 = im_string(data);
		    str2 = im_string(data);
		    str3 = im_string(data);
		    if (str3) {
			ucase(str3);
			mm_option((SUPPORTSTREAM *) data->parentstruct,
				  str1, str2,
				  strcmp("[READ-ONLY]", str3) != 0);
			break;
		    }
		    sprintf(reply->tmp,
			    "Incorrect arguments to keyword: %s",
			    reply->key);
		    mm_log(reply->tmp, WARN);
		    break;
		case IMKEY_MYRIGHTS:
		    str1 = im_string(data);
		    str2 = im_string(data);
		    str3 = im_string(data);
		    ucase(str1);
		    if (!strcmp(str1, "ADDRESSBOOK")) {
			imsp_abookacl((SUPPORTSTREAM *) data->parentstruct,
				      str2, NULL, str3);
		    }
		    break;
		case IMKEY_ACL:
		    str1 = im_string(data);
		    str2 = im_string(data);
		    str3 = im_string(data);
		    str4 = im_string(data);
		    ucase(str1);
		    if (!strcmp(str1, "ADDRESSBOOK")) {
			imsp_abookacl((SUPPORTSTREAM *) data->parentstruct,
				      str2, str3, str4);
		    }
		    break;
		default:
		    sprintf(reply->tmp, "Unhandled unsolicited keyword: %s",
			    reply->key);
		    mm_log(reply->tmp, WARN);
		    break;
	    }
	} else if (tag && (!strcmp(reply->tag, tag)
			   || !strcmp(reply->tag, "+"))) {
	    return (reply->keyval == IMKEY_OK || *reply->tag == '+');
	} else {
	    sprintf(reply->tmp,
		    "Unexpected tagged response: %.80s %.80s %.80s",
		    reply->tag, reply->key, reply->text);
	    mm_log(reply->tmp, WARN);
	}
    }

    return (NIL);
}

/* output an IMAP/IMSP string
 *  data->tcp    -- pointer to current tcp stream
 *  data->reply  -- parsed reply buffer to be used
 *  data->gensym -- generated symbol value
 *  str          -- an sprintf style string with the following meanings:
 *            %a -- atom
 *            %s -- astring (will be quoted or literalized as needed)
 *            %S -- same as %s, but with security (zeros temp space)
 *            %n -- nstring (quoted, will be literalized as needed)
 *            %d -- decimal
 *            %t -- do tag
 *            %k -- do key-value pair list
 *            %% -- %
 */
long im_send(IMSTREAMDATA *data, char *str, ...)
{
    va_list ap;
    char *wkspace, *wkptr, *astr, *scan, *istr;
    char c;
    int wksize, wkused, len, result, litpos, i, didtag, secureflag;
    IMPARSEDREPLY *reply;
    keyvalue *kv;
    long val, j;
    char tagbuf[10];

    /* initialize argument list */
    va_start(ap, str);

    /* initialize workspace */
    if (!data) {
	strcpy(tagbuf, "*");
    } else {
	if (!data->tcp) return (NIL);
	sprintf(tagbuf, "A%05ld", data->gensym - 1);
	reply = &data->reply;
    }
    didtag = 0;
    wkused = 0;
    secureflag = 0;
    wksize = (strlen(str) + 1) * 4 + 512;
    wkptr = wkspace = fs_get(wksize);

    /* start copying string */
    while (*str) {
	wkused = wkptr - wkspace;
	if (*str == '%' && *++str != '%') {
	    /* calculate max length needed for argument */
	    len = 0;
	    switch (*str) {
		case 't':
		    sprintf(tagbuf, "A%05ld", data->gensym++);
		    astr = tagbuf;
		    len = strlen(tagbuf);
		    break;
		case 'a':
		    astr = va_arg(ap, char *);
		    len = strlen(astr);
		    break;
		case 'S':
		    secureflag = 1;
		case 's':
		case 'n':
		    astr = va_arg(ap, char *);
		    len = strlen(astr) + MAX_LITERAL_EXTRA;
		    val = 1;
		    break;
		case 'd':
		    val = va_arg(ap, long);
		    len = MAX_INTEGER_LEN;
		    break;
		case 'k':
		    val = va_arg(ap, long);
		    kv = va_arg(ap, keyvalue *);
		    for (len = 0, j = 0; j < val; ++j) {
			len += 2 + strlen(kv[j].key) + strlen(kv[j].value);
		    }
		    val *= 2;
		    len += MAX_LITERAL_EXTRA * val;
		    j = 0;
		    break;
	    }

	    /* grow workspace if needed */
	    if (wkused + len >= wksize - 1) {
		fs_resize((void **) &wkspace, wksize += len + strlen(str));
		wkptr = wkspace + wkused;
	    }

	    /* copy argument */
	    switch (*str) {
		case 'a':
		case 't':
		    strcpy(wkptr, astr);
		    wkptr += len;
		    break;
		case 's':
		case 'S':
		case 'n':
		case 'k':
		    len -= MAX_LITERAL_EXTRA;
		    for (j = 0; j < val; ++j) {
			/* get string to use */
			if (*str == 'k') {
			    astr = (j & 1) ? kv[j / 2].value : kv[j / 2].key;
			    len = strlen(astr);
			    *wkptr++ = ' ';
			}
			scan = astr;

			/* send empty string as nil or "" */
			if (!*scan) {
			    if (*str == 'n') {
				*wkptr++ = 'n';
				*wkptr++ = 'i';
				*wkptr++ = 'l';
			    } else {
				*wkptr++ = '"';
				*wkptr++ = '"';
			    }
			}

			/* try an atom */
			if (*str != 'n' && *scan) {
			    while (*scan && isatom(*scan)) ++scan;
			    if (*scan) {
				scan = astr;
			    } else {
				strcpy(wkptr, astr);
				wkptr += len;
			    }
			}

			/* try a quoted string */
			if (*scan) {
			    while (*scan && isqstr(*scan)) ++scan;
			    if (*scan) {
				scan = astr;
			    } else {
				*wkptr++ = '"';
				strcpy(wkptr, astr);
				wkptr += len;
				*wkptr++ = '"';
			    }
			}

			/* send a literal */
			if (*scan) {
			    sprintf(wkptr, "{%d}\015\012", len);
			    wkptr += strlen(wkptr);
			    if (!data) {
				printf("%s", wkspace);
			    } else {
				tcp_sout(data->tcp, wkspace, wkptr - wkspace);
			    }
			    if (secureflag) {
				memset(wkspace, '\0', wkptr - wkspace);
			    }
			    if (data) {
				if (!im_reply(data, tagbuf)) {
				    fs_give((void **) &wkspace);
				    return (NIL);
				}
				if (*reply->tag != '+') {
				    reply->text =
					"Literal prompt expected from server";
				    fs_give((void **) &wkspace);
				    return (NIL);
				}
			    }
			    strcpy(wkspace, astr);
			    wkptr = wkspace + len;
			}
		    }
		    break;
		case 'd':
		    sprintf(wkptr, "%ld", val);
		    wkptr += strlen(wkptr);
		    break;
	    }
	    ++str;
	} else {
	    /* make extra space if needed */
	    if (wkused == wksize - 1) {
		fs_resize((void **) &wkspace, wksize *= 2);
		wkptr = wkspace + wkused;
	    }

	    /* copy character */
	    *wkptr++ = *str++;
	}
    }
    va_end(ap);
    wkused = wkptr - wkspace;
    if (!data) {
	wkspace[wkused] = '\0';
	printf("%s", wkspace);
    } else {
	tcp_sout(data->tcp, wkspace, wkused);
    }
    if (secureflag) memset(wkspace, '\0', wkused);
    fs_give((void **) &wkspace);

    return (data ? im_reply(data, tagbuf) : 1);
}
