/* imsptest.c -- test Interactive Mail Support Protocol routines
 *
 *	(C) Copyright 1993-1994 by Carnegie Mellon University
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Author: Chris Newman
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "mail.h"
#include "osdep.h"
#include "imap2.h"
#include "imutil.h"
#include "support.h"
#include "imsp.h"

/* if on unix, you may wish to set this to 1 to test local options
 */
#define TRYLOCAL 0
#if TRYLOCAL
#include "unixfile.h"
#endif

extern DRIVER imapdriver;

char *curhst = NIL;		/* currently connected host */
char *curusr = NIL;		/* current login user */

/* Prompt user for input
 * Accepts: pointer to prompt message
 *          pointer to input buffer
 *          max length of input
 */

void prompt (char *msg,char *txt, long max)
{
  printf ("%s",msg);
  if (fgets (txt, max, stdin) && *txt) {
      txt[strlen(txt) - 1] = '\0';
  }
}

/* Interfaces to C-client */


void mm_searched (MAILSTREAM *stream,long number)
{
}


void mm_exists (MAILSTREAM *stream,long number)
{
}


void mm_expunged (MAILSTREAM *stream,long number)
{
}


void mm_flags (MAILSTREAM *stream,long number)
{
}


void mm_notify (MAILSTREAM *stream,char *string,long errflg)
{
  mm_log (string,errflg);
}


void mm_mailbox (char *string)
{
  putchar (' ');
  puts (string);
}


void mm_bboard (char *string)
{
  putchar (' ');
  puts (string);
}

void mm_log (char *string,long errflg)
{
  switch ((short) errflg) {
  case NIL:
    printf ("[%s]\n",string);
    break;
  case PARSE:
  case WARN:
    printf ("%%%s\n",string);
    break;
  case ERROR:
    printf ("?%s\n",string);
    break;
  }
}


void mm_dlog (char *string)
{
  puts (string);
}


void mm_login (char *host,char *user,char *pwd,long trial)
{
    char tmp[MAILTMPLEN];
    char *tmppass;
  
    if (curhst) fs_give ((void **) &curhst);
    curhst = (char *) fs_get (1+strlen (host));
    strcpy (curhst,host);
    sprintf (tmp,"{%s} username: ",host);
    prompt (tmp,user,MAILTMPLEN);
    if (curusr) fs_give ((void **) &curusr);
    curusr = (char *) fs_get (1+strlen (user));
    strcpy (curusr,user);
#if 0
    prompt ("password: ",pwd,MAILTMPLEN);
#else
    tmppass = (char *) getpass("password: ");
    strcpy(pwd, tmppass);
    memset(tmppass, '\0', strlen(tmppass));
#endif
}


void mm_critical (MAILSTREAM *stream)
{
}


void mm_nocritical (MAILSTREAM *stream)
{
}


long mm_diskerror (MAILSTREAM *stream,long errcode,long serious)
{
#if UNIXLIKE
  kill (getpid (),SIGSTOP);
#else
  abort ();
#endif
  return NIL;
}


void mm_fatal (char *string)
{
  printf ("?%s\n",string);
}


void mm_addressbook(SUPPORTSTREAM *stream, char *abook_name, char sep_char)
{
    printf("address book: %s\n", abook_name);
}


void mm_address(abook *ab, char *name, keyvalue *kv, int count)
{
    int j;
    
    printf("entry name: %s\n", name);
    for (j = 0; j < count; ++j) {
	printf("  %s: %s\n", kv[j].key, kv[j].value);
    }
}

void mm_abookacl(abook *ab, char *identifier, char *rights)
{
    if (identifier) {
	printf("ACL: %s %s\n", identifier, rights);
    } else {
	printf("MYRIGHTS: %s\n", rights);
    }
}

void mm_option(SUPPORTSTREAM *s, char *option, char *value,
	       int read_write_flag)
{
    printf("option: %s = %s rw=%d\n", option, value, read_write_flag);
}

/* main program
 */
main(int argc, char **argv)
{
    SUPPORTSTREAM *stream;
    abook *ab;
    keyvalue kv[5];
    int remote;
    char oldname[128], newname[128];

    /* link in IMAP driver */
    mail_link(&imapdriver);
    if (argc != 2) {
	fprintf(stderr, "usage: imsptest <hostname>\n");
	exit(1);
    }

    /* do once locally, and once remotely */
    remote = 0;
    do {
	/* create a new support stream */
	stream = support_new();

#if TRYLOCAL
	if (!remote) {
	    /* add local unix file driver */
	    support_add_odriver(stream, &unixfile_option_drvr);
	    support_add_adriver(stream, &unixfile_abook_drvr);
	} else
#else
	    remote = 1;
#endif
	{
	    /* open connection to IMSP server */
	    if (!imsp_open(stream, argv[1])) {
		fprintf(stderr, "failed to open host: %s\n", argv[1]);
		exit(1);
	    }
	}

	printf("- Creating test options\n");
	if (option_lock(stream, "test.one")) {
	    option_set(stream, "test.one", "foo1");
	    option_unlock(stream, "test.one");
	}
	if (option_lock(stream, "test.two")) {
	    option_set(stream, "test.two", "foo\"bar");
	    option_unlock(stream, "test.two");
	}
	if (option_lock(stream, "test.three")) {
	    option_set(stream, "test.three", "foo\nbar");
	    option_unlock(stream, "test.three");
	}

	printf("- Display test options\n");
	option_get(stream, "test.*");

	printf("- Display IMSP options\n");
	option_get(stream, "imsp.*");

	printf("- Remove test options\n");
	if (option_lock(stream, "test.one")) {
	    option_set(stream, "test.one", NULL);
	    option_unlock(stream, "test.one");
	}
	if (option_lock(stream, "test.two")) {
	    option_set(stream, "test.two", NULL);
	    option_unlock(stream, "test.two");
	}
	if (option_lock(stream, "test.three")) {
	    option_set(stream, "test.three", NULL);
	    option_unlock(stream, "test.three");
	}

	printf("- Look for any remaining test options\n");
	option_get(stream, "test.*");

	printf("- List available address books\n");
	abook_find(stream, "*");

	printf("- Open default address book\n");
	ab = abook_open(stream, NULL, NULL);
	sprintf(oldname, "%s.foo", abook_name(ab));

	printf("- Store test entries in address book\n");
	kv[0].key = "email";
	kv[0].value = "chrisn+@cmu.edu";
	kv[1].key = "phone";
	kv[1].value = "(412) 268-6403";
	kv[2].key = "fax";
	kv[2].value = "(412) 268-4987";
	if (abook_lock(ab, "Chris Newman")) {
	    abook_store(ab, "Chris Newman", kv, 3);
	    abook_unlock(ab, "Chris Newman");
	}
	kv[0].value = "chrisn+@cmu.edu\015\012jgm+@cmu.edu";
	if (abook_lock(ab, "CMU IMSP Team")) {
	    abook_store(ab, "CMU IMSP Team", kv, 1);
	    abook_unlock(ab, "CMU IMSP Team");
	}
	kv[0].value = "foo@bar.baz";
	kv[1].value = "(000) 000-0000";
	if (abook_lock(ab, "Foo Bar")) {
	    abook_store(ab, "Foo Bar", kv, 2);
	    abook_unlock(ab, "Foo Bar");
	}

	printf("- Show list of address book entries\n");
	abook_getlist(ab);

	printf("- Show entries with chrisn+\n");
	kv[0].value = "*chrisn+*";
	abook_search(ab, "*", kv, 1);

	printf("- Fetch two entries\n");
	abook_fetch(ab, "Chris Newman");
	abook_fetch(ab, "Foo Bar");

	printf("- Expand CMU IMSP team:\n%s\n", abook_expand(ab, "CMU IMSP Team"));

	printf("- Show ACL and MYRIGHTS on this address book\n");
	abook_getacl(ab, 0);
	abook_getacl(ab, 1);
    
	printf("- Remove entires\n");
	if (abook_lock(ab, "Chris Newman")) {
	    abook_deleteent(ab, "Chris Newman");
	    abook_unlock(ab, "Chris Newman");
	}
	if (abook_lock(ab, "CMU IMSP Team")) {
	    abook_deleteent(ab, "CMU IMSP Team");
	    abook_unlock(ab, "CMU IMSP Team");
	}
	if (abook_lock(ab, "Foo Bar")) {
	    abook_deleteent(ab, "Foo Bar");
	    abook_unlock(ab, "Foo Bar");
	}

	printf("- Show list of remaining address book entries\n");
	abook_getlist(ab);

	/* close address book */
	abook_close(ab);

	/* create address book */
	abook_create(stream, oldname);
	
	printf("- List available address books\n");
	abook_find(stream, "*");

	/* rename address book */
	sprintf(newname, "%sbar", oldname);
	abook_rename(stream, oldname, newname);
	
	printf("- List available address books\n");
	abook_find(stream, "*");

	/* delete new address book */
	abook_delete(stream, newname);

	printf("- List available address books\n");
	abook_find(stream, "*");
    } while (!remote++);

    /* clean up */
    support_done(stream);
    exit(0);
}
