/* imsp.c -- Interactive Mail Support Protocol drivers
 *
 *	(C) Copyright 1993-1994 by Carnegie Mellon University
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Author: Chris Newman <chrisn+@cmu.edu>
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include "mail.h"
#include "osdep.h"
#include "imap2.h"
#include "imutil.h"
#include "support.h"
#include "imsp.h"
#include "misc.h"
#include "rfc822.h"
#include "acte.h"

extern struct acte_client *login_acte_client[];

/* driver name for IMSP drivers */
static char drvrname[] = "IMSP";

/* flag indicating we're in an expand operation
 */
static int in_expand = 0;

/* IMSP addressbook function prototypes
 */
long imsp_abook_create(SUPPORTSTREAM *, char *);
long imsp_abook_delete(SUPPORTSTREAM *, char *);
long imsp_abook_rename(SUPPORTSTREAM *, char *, char *);
long imsp_abook_find(SUPPORTSTREAM *, char *);
abook *imsp_abook_open(SUPPORTSTREAM *, char *);
void imsp_abook_close(abook *);
long imsp_abook_getlist(abook *);
long imsp_abook_search(abook *, char *, keyvalue *, long);
long imsp_abook_fetch(abook *, char *);
long imsp_abook_lock(abook *, char *);
void imsp_abook_unlock(abook *, char *);
long imsp_abook_store(abook *, char *, keyvalue *, long);
long imsp_abook_deleteent(abook *, char *);
char *imsp_abook_expand(abook *, char *);
long imsp_abook_getacl(abook *, int);
long imsp_abook_setacl(abook *, char *, char *);
void imsp_abook_pfree(struct abookdriver *);

/* IMSP option function prototypes
 */
long imsp_option_get(SUPPORTSTREAM *, char *);
long imsp_option_lock(SUPPORTSTREAM *, char *);
void imsp_option_unlock(SUPPORTSTREAM *, char *);
long imsp_option_set(SUPPORTSTREAM *, char *, char *);
void imsp_option_pfree(struct optiondriver *);


/* IMSP addressbook driver
 */
static abookdriver IMSP_abook_drvr = {
    drvrname,
    NULL, NULL,
    imsp_abook_create,
    imsp_abook_delete,
    imsp_abook_rename,
    imsp_abook_find,
    imsp_abook_open,
    imsp_abook_close,
    imsp_abook_getlist,
    imsp_abook_search,
    imsp_abook_fetch,
    imsp_abook_lock,
    imsp_abook_unlock,
    imsp_abook_store,
    imsp_abook_deleteent,
    imsp_abook_expand,
    imsp_abook_getacl,
    imsp_abook_setacl,
    imsp_abook_pfree
};

/* IMSP option driver
 */
static optiondriver IMSP_option_drvr = {
    drvrname,
    NULL, NULL,
    imsp_option_get,
    imsp_option_lock,
    imsp_option_unlock,
    imsp_option_set,
    imsp_option_pfree
};

/* open an IMSP connection and install IMSP option/address-book drivers
 *  returns: T on success, NIL on failure
 */
long imsp_open(SUPPORTSTREAM *s, char *host)
{
    long map_maxlogintrials = MAXLOGINTRIALS;
    char username[MAILTMPLEN], pwd[MAILTMPLEN];
    long i, j;
    IMPARSEDREPLY *reply = &s->imsp.reply;
    abookdriver *advr;
    optiondriver *odvr;
    struct acte_client **mech;
    void *state;
    char *in, *out, *scan;
    int r, gotdata, rtval;
    char *(*func)();
    
    /* open the connection */
    if (!s || (s->imsp.tcp = tcp_open(host, "imsp", IMSP_PORT)) == NULL) {
	return (NIL);
    }

    /* get the reply */
    if (im_reply(&s->imsp, NULL)) {
	if (reply->keyval == IMKEY_PREAUTH) {
	    advr = (abookdriver *) fs_get(sizeof (abookdriver));
	    *advr = IMSP_abook_drvr;
	    support_add_adriver(s, advr);
	    odvr = (optiondriver *) fs_get(sizeof (optiondriver));
	    *odvr = IMSP_option_drvr;
	    support_add_odriver(s, odvr);
	    return (T);
	} else {
	    for (mech = login_acte_client; *mech; mech++) {
		if ((*mech)->start(tcp_host (s->imsp.tcp), 0,
				   ACTE_PROT_NONE, 0, 0, 0, &state)) continue;
		rtval = im_send(&s->imsp, "%t AUTHENTICATE %s\015\012",
			     (*mech)->auth_type);
		r = gotdata = 0;
		while (rtval || (reply->keyval != IMKEY_BYE && s->imsp.tcp)) {
		    if (strcmp(reply->tag, "+")) break;
		    in = rfc822_base64(reply->text, strlen(reply->text), &i);
		    if (!in || (r = (*mech)->auth(state, i, in, &j, &out))
			== ACTE_FAIL) {
			if (in) fs_give((void **) &in);
			/* Force a BAD */
			rtval = im_send(&s->imsp, "*\015\012");
		    } else {
			if (i) gotdata = 1;
			fs_give((void **) &in);
			in = rfc822_binary(out, j, &i);
			/* Removing line breaks from base64 string */
			for (scan = in, out = in; *scan; ++scan) {
			    if (*scan != '\012' && *scan != '\015')
				*out++ = *scan;
			}
			*out = '\0';
			rtval = im_send(&s->imsp, "%a\015\012", in);
			fs_give((void **) &in);
		    }
		}
		if (!strcmp(reply->key, "OK")) {
		    if (r == ACTE_DONE) {
			/* we need to know the user name for address books */
			(*mech)->query_state(state, &out, &i, &func, &func,
					     &i);
			s->imsp.username = cpystr(out);
			(*mech)->free_state(state);
			advr = (abookdriver *) fs_get(sizeof (abookdriver));
			*advr = IMSP_abook_drvr;
			support_add_adriver(s, advr);
			odvr = (optiondriver *) fs_get(sizeof (optiondriver));
			*odvr = IMSP_option_drvr;
			support_add_odriver(s, odvr);
			return (T);
		    }
		    mm_log("Server ended authentication exchange too early",
			   ERROR);
		}
		(*mech)->free_state(state);
		if (gotdata) break;
	    }
	    if (!*mech) for (i = 0; i < map_maxlogintrials; ++i) {
		*pwd = 0;
		mm_login(tcp_host(s->imsp.tcp), username, pwd, i);
		if (!*pwd) {
		    mm_log("Login aborted", ERROR);
		    break;
		} else if (im_send(&s->imsp, "%t LOGIN %s %S\015\012",
				   username, pwd)) {
		    memset(pwd, '\0', strlen(pwd));
		    s->imsp.username = cpystr(username);
		    advr = (abookdriver *) fs_get(sizeof (abookdriver));
		    *advr = IMSP_abook_drvr;
		    support_add_adriver(s, advr);
		    odvr = (optiondriver *) fs_get(sizeof (optiondriver));
		    *odvr = IMSP_option_drvr;
		    support_add_odriver(s, odvr);
		    return (T);
		} else {
		    mm_log(reply->text, WARN);
		    if (reply->keyval == IMKEY_BYE || !s->imsp.tcp) break;
		}
	    }
	    if (i >= map_maxlogintrials) {
		mm_log("Too many login failures", ERROR);
	    }
	}
    } else {
	mm_log(reply->text, ERROR);
    }

    /* clean up on error exit */
    if (s->imsp.tcp) {
	tcp_close(s->imsp.tcp);
	s->imsp.tcp = NULL;
    }
    
    return (NIL);
}

static long dolog(SUPPORTSTREAM *s, long result)
{
    if (result == NIL) {
	mm_log(s->imsp.reply.text, ERROR);
    }

    return (result);
}

long imsp_abook_create(SUPPORTSTREAM *s, char *name)
{
    return (dolog(s, im_send(&s->imsp,
			     "%t CREATEADDRESSBOOK %s\015\012", name)));
}

long imsp_abook_delete(SUPPORTSTREAM *s, char *name)
{
    return (dolog(s, im_send(&s->imsp,
			     "%t DELETEADDRESSBOOK %s\015\012", name)));
}

long imsp_abook_rename(SUPPORTSTREAM *s, char *oldname, char *newname)
{
    return (dolog(s, im_send(&s->imsp, "%t RENAMEADDRESSBOOK %s %s\015\012",
			     oldname, newname)));
}

long imsp_abook_find(SUPPORTSTREAM *s, char *pattern)
{
    return (dolog(s, im_send(&s->imsp,
			     "%t ADDRESSBOOK %s\015\012", pattern)));
}

abook *imsp_abook_open(SUPPORTSTREAM *s, char *name)
{
    abook *ab;

    if (!name && !(name = s->imsp.username)) return (NULL);
    ab = (abook *) fs_get(sizeof (abook) + strlen(name) + 1);
    ab->name = (char *) (ab + 1);
    strcpy(ab->name, name);

    return (ab);
}

void imsp_abook_close(abook *ab)
{
    fs_give((void **) &ab);
}

long imsp_abook_getlist(abook *ab)
{
    ab->parent->activeab = ab;
    return (dolog(ab->parent,
		  im_send(&ab->parent->imsp, "%t SEARCHADDRESS %s\015\012",
			  ab->name)));
}

long imsp_abook_search(abook *ab, char *pattern, keyvalue *kv, long count)
{
    ab->parent->activeab = ab;
    if (!strcmp("*", pattern)) {
	return (dolog(ab->parent, im_send(&ab->parent->imsp,
					  "%t SEARCHADDRESS %s%k\015\012",
					  ab->name, count, kv)));
    }
    return (dolog(ab->parent, im_send(&ab->parent->imsp,
				      "%t SEARCHADDRESS %s name %s%k\015\012",
				      ab->name, pattern, count, kv)));
}

long imsp_abook_lock(abook *ab, char *name)
{
    ab->parent->activeab = ab;
    return (dolog(ab->parent, im_send(&ab->parent->imsp,
				      "%t LOCK ADDRESSBOOK %s %s\015\012",
				      ab->name, name)));
}

void imsp_abook_unlock(abook *ab, char *name)
{
    (void) dolog(ab->parent, im_send(&ab->parent->imsp,
				     "%t UNLOCK ADDRESSBOOK %s %s\015\012",
				     ab->name, name));
}

long imsp_abook_store(abook *ab, char *name, keyvalue *kv, long count)
{
    ab->parent->activeab = ab;
    return (dolog(ab->parent, im_send(&ab->parent->imsp,
				      "%t STOREADDRESS %s %s%k\015\012",
				      ab->name, name, count, kv)));
}

long imsp_abook_deleteent(abook *ab, char *name)
{
    return (dolog(ab->parent, im_send(&ab->parent->imsp,
				      "%t DELETEADDRESS %s %s\015\012",
				      ab->name, name)));
}

long imsp_abook_fetch(abook *ab, char *name)
{
    ab->parent->activeab = ab;
    return (dolog(ab->parent, im_send(&ab->parent->imsp,
				      "%t FETCHADDRESS %s %s\015\012",
				      ab->name, name)));

}

/* storage for email expand function
 */
static struct memlist {
    struct memlist *next;
    char *abook, *entry, *data;
} *members_list;
static struct elist {
    struct elist *next;
    int len;
    char str[1];
} *email_list;
static int email_len, email_count;
static int member_add(char *abook, char *entry, char *data)
{
    struct memlist **pmem, *mem;

    for (pmem = &members_list; (mem = *pmem); pmem = &mem->next) {
	if (!strcmp(entry, mem->entry) && !strcmp(abook, mem->abook)) {
	    return (1);
	}
    }
    mem = *pmem = (struct memlist *) malloc(sizeof (struct memlist));
    if (!mem) return (-1);
    mem->abook = abook;
    mem->entry = entry;
    mem->data = data;
    mem->next = NULL;
    return (0);
}
static void members_parse(char *members, char *parent)
{
    char *abook, *entry, *end, *data, *dst;
    
    if (members) {
	data = end = members;
	do {
	    abook = entry = dst = end;
	    while (*end && (end[0] != '\015' && end[1] != '\012')) {
		if (*end == '\\') {
		    *dst++ = *++end;
		    ++end;
		} else if (*end == ':' && entry == abook) {
		    *dst++ = '\0';
		    entry = ++end;
		} else {
		    *dst++ = *end++;
		}
	    }
	    *dst = '\0';
	    if (*end) end += 2;
	    if (abook == entry) abook = parent;
	    if (member_add(abook, entry, data) == 0) data = NULL;
	} while (*end);
	if (data) free(data);
    }
}
static int email_add(char *email, int len)
{
    struct elist **pptr, *ptr;

    for (pptr = &email_list; (ptr = *pptr); pptr = &ptr->next) {
	if (ptr->len == len && !strncmp(ptr->str, email, len)) return (0);
    }
    ptr = *pptr = (struct elist *) malloc(sizeof (struct elist) + len);
    if (!ptr) return (-1);
    strncpy(ptr->str, email, len);
    ptr->str[len] = '\0';
    ptr->len = len;
    ptr->next = NULL;
    email_len += len;
    ++email_count;
    return (0);
}

char *imsp_abook_expand(abook *ab, char *name)
{
    static char *expanded = NULL;
    struct memlist *mem, *mnext;
    struct elist *eptr, *enext;
    char *pos;

    /* initialize */
    if (expanded) {
	free(expanded);
	expanded = NULL;
    }
    in_expand = 1;
    email_list = NULL;
    email_len = email_count = 0;
    members_list = NULL;
    member_add(ab->name, name, NULL);

    /* walk through all interesting address entries */
    for (mem = members_list; mem; mem = mem->next) {
	im_send(&ab->parent->imsp, "%t FETCHADDRESS %s %s\015\012",
		mem->abook, mem->entry);
    }
    in_expand = 0;

    /* free address book entry list */
    for (mem = members_list; mem; mem = mnext) {
	mnext = mem->next;
	if (mem->data) free(mem->data);
	free((char *) mem);
    }

    /* copy email addresses into complete list & free separate addresses */
    pos = expanded = malloc(email_len + email_count * 2 + 1);
    for (eptr = email_list; eptr; eptr = enext) {
	enext = eptr->next;
	if (pos) {
	    strcpy(pos, eptr->str);
	    pos += eptr->len;
	    *pos++ = '\015';
	    *pos++ = '\012';
	}
	free((char *) eptr);
    }
    if (pos) pos[pos == expanded ? 0 : -2] = '\0';
    
    return (expanded);
}

/* fetchaddress callback
 */
void imsp_fetchaddress(SUPPORTSTREAM *s, char *abname, char *name,
		       keyvalue *kv, long kvcount)
{
    int i;
    char *pos, *end;
    
    if (kvcount) {
	if (in_expand) {
	    /* look for "email" field */
	    for (i = 0; i < kvcount; ++i) {
		lcase(kv[i].key);
		if (kv[i].value) {
		    if (!strcmp(kv[i].key, "email")) {
			pos = end = kv[i].value;
			while (*end) {
			    if (end[0] == '\015' && end[1] == '\012') {
				email_add(pos, end - pos);
				pos = (end += 2);
			    } else {
				++end;
			    }
			}
			email_add(pos, end - pos);
		    } else if (!strcmp(kv[i].key, "members")) {
			members_parse(cpystr(kv[i].value), abname);
		    }
		}
	    }
	} else {
	    mm_address(s->activeab, name, kv, kvcount);
	}
    }
}

long imsp_abook_getacl(abook *ab, int myrights)
{
    ab->parent->activeab = ab;
    return (dolog(ab->parent, im_send(&ab->parent->imsp,
				      "%t %s ADDRESSBOOK %s\015\012",
				      myrights ? "MYRIGHTS" : "GETACL",
				      ab->name)));
}

long imsp_abook_setacl(abook *ab, char *identifier, char *rights)
{
    ab->parent->activeab = ab;
    return (dolog(ab->parent, im_send(&ab->parent->imsp,
				      "%t SETACL ADDRESSBOOK %s %s %s\015\012",
				      ab->name, identifier, rights)));
}

/* abookacl callback
 */
void imsp_abookacl(SUPPORTSTREAM *s, char *abname, char *identifier,
		   char *rights)
{
    mm_abookacl(s->activeab, identifier, rights);
}

void imsp_abook_pfree(struct abookdriver *adrvr)
{
    fs_give((void **) &adrvr);
    return;
}

long imsp_option_get(SUPPORTSTREAM *s, char *pattern)
{
    return (dolog(s, im_send(&s->imsp, "%t GET %s\015\012", pattern)));
}

long imsp_option_lock(SUPPORTSTREAM *s, char *name)
{
    return (dolog(s, im_send(&s->imsp, "%t LOCK OPTION %s\015\012", name)));
}

void imsp_option_unlock(SUPPORTSTREAM *s, char *name)
{
    dolog(s, im_send(&s->imsp, "%t UNLOCK OPTION %s\015\012", name));
}

long imsp_option_set(SUPPORTSTREAM *s, char *name, char *value)
{
    if (value) {
	return (dolog(s, im_send(&s->imsp, "%t SET %s %s\015\012",
				 name, value)));
    }
    return (dolog(s, im_send(&s->imsp, "%t UNSET %s\015\012", name)));
}

void imsp_option_pfree(struct optiondriver *odrvr)
{
    fs_give((void **) &odrvr);
    return;
}
