(* 
 * Copyright (c) 2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *)

(* ACAP throttles.

   A throttle regulates how much data can be sent through.  There are
   two types, a hard throttle and a soft throttle.

   Hard throttles queue up data making sure it is less then the
   specified number.  If more occur it raises WAYTOOMANY.

   Soft throttles queue up data making sure that if more than the
   specified number are returned, only the first so many go through,
   and raising TOOMANY. 

   The user of the throttles must know what to do with the return value.

   Note that throttles inherently have state; the implementation may
   NOT be thread safe, so beware of calling from multiple threads!

   *)

signature THROTTLE =
sig
  type 'a throttle

  datatype hardresult = H_WAYTOOMANY | H_OK
  datatype softresult = S_TOOMANY of int | S_OK
  datatype result = OK

  (* create a hard throttle, plus what to call when done *)
  val createhard : int -> ('a -> unit) -> (hardresult -> unit) -> 'a throttle

  (* create a soft throttle; the first number is the limit *)
  val createsoft : (int * int) -> ('a -> unit) -> (softresult -> unit) -> 
                   'a throttle

  (* create a sorting throttle *)
  val createsort : (('a * 'a) -> order) -> ('a -> unit) -> 
                   (result -> unit) -> 'a throttle

  (* a throttle that just passes through data *)
  val createpass : ('a -> unit) -> (result -> unit) -> 'a throttle

  (* send data into the throttle *)
  val call : 'a throttle -> 'a -> unit

  (* indicates that no more data will be coming through the throttle;
     if the limit is exceeded it just throws out the data *)
  val done : 'a throttle -> unit

end
