(* 
 * Copyright (c) 2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *)

structure Entry :> ENTRY =
struct
  structure Dict = ListMapFn(type ord_key = String.string
			     val compare = String.compare)

  type attribute = {acl : Acl.acl, attribute : string,
		    size : int list, value : Value.value}
  type attribute' = {acl : Acl.acl option, attribute : string,
		     size : int list, value : Value.value}
  type storedata = {acl : Acl.acl option, attribute : string, 
		    value : Value.value option}
  (* fully qualified acls *)
  datatype entry = NIL of Value.value | E of attribute Dict.map
  (* incomplete acls *)
  datatype entry' = NIL' of Value.value | E' of attribute' Dict.map

  (* raised when an attempt to store an illegal value is made to this
   dataset---returns attribute raised on *)
  exception EnforcedValidation of (string * string)

  (* raised when an attempt is made to store with a UNCHANGEDSINCE;
     returns the time it was last modified *)
  exception Modified of AcapTime.acaptime

  (* renames aren't implemented *)
  exception RenameNotImpemented

  (* ACL problem *)
  exception NotPermitted
  
  (* raised when entry file is inconsistent *)
  exception BadEntryFile
  exception InternalError of string

  (* create a blank attribute *)
  fun makeNull (attr, acl) = 
    {acl=acl, attribute=attr, size=nil, value=Value.Nil}

  fun checkchar c rdr t =
    case rdr t of
      NONE => raise BadEntryFile
    | (SOME (c', t)) => if c <> c' then raise BadEntryFile
			else t

  (* val scanAttribute : (char, 'a) reader -> (attribute, 'a) reader *)
  fun scanAttr getc =
    let
      val scanOpen = ParserComb.string "<attr>"
      val scanName = ParserComb.token (fn c => c <> #"*")
      val scanStar = ParserComb.char #"*"
      val scanAclOpt = ParserComb.option Acl.scan
      val scanValue = Value.scan
      val scanClose = ParserComb.string "</attr>"

      val scan = ParserComb.seqWith #2 (scanOpen, scanName)
      val scan = ParserComb.seqWith #1 (scan, scanStar)
      val scan = ParserComb.seq (scan, scanAclOpt)
      val scan = ParserComb.seqWith (#1) (scan, scanStar)
      val scan = ParserComb.seqWith 
                     (fn ((attrname, acl), value) =>
		      {attribute=attrname,
		       acl=acl,
		       value=value,
		       size=Value.size value}) (scan, scanValue)
      val scan = ParserComb.seqWith #1 (scan, scanClose)
    in
      scan getc
    end

  fun scanEntry getc =
    let
      val scanOpen1 = ParserComb.string "<entry name=\""
      val scanName = ParserComb.wrap 
	(ParserComb.option (ParserComb.token (fn c => c <> #"\"")),
	 (fn NONE => "" | SOME s => s))
					     
      val scanOpen2 = ParserComb.string "\">"

      val scanOpen = ParserComb.seqWith #2 (scanOpen1, scanName)
      val scanOpen = ParserComb.seqWith #1 (scanOpen, scanOpen2)

      val scanClose = ParserComb.skipBefore Char.isSpace
			    (ParserComb.string "</entry>")
      val scanAttr = ParserComb.skipBefore Char.isSpace scanAttr
			    
      fun scanAttrs dict NONE = ParserComb.result dict
	| scanAttrs dict (SOME attr) =
	ParserComb.bind (ParserComb.option scanAttr,
			 scanAttrs (Dict.insert (dict, #attribute attr, attr)))

      val scanAttrs = ParserComb.bind (ParserComb.wrap (scanAttr, SOME), 
				       scanAttrs Dict.empty)

      val scan = ParserComb.seq (scanOpen, 
				 ParserComb.wrap (scanAttrs, E'))
      val scan = ParserComb.seqWith #1 (scan, scanClose)
    in
      scan getc
    end

  fun scanNil getc =
    let
      val scanOpen = ParserComb.string "<nil name=\""
      val scanName = ParserComb.token (fn c => c <> #"\"")
      val scanClose = ParserComb.string "\"></nil>"

      val scan = ParserComb.seqWith #2 (scanOpen, scanName)
      val scan = ParserComb.wrap (scan, (fn x => (x, NIL' (Value.Single x))))
      val scan = ParserComb.seqWith #1 (scan, scanClose)
    in
      scan getc
    end

  fun scan getc = ParserComb.or (scanEntry, scanNil) getc

  (* format an entry to save *)
  fun fmt (name, E' e) =
    let
      fun fmtAttr {acl, attribute, size, value} =
	"<attr>" ^
	attribute ^ "*" ^ (case acl of NONE => "*" | (SOME acl) => Acl.fmt acl)
	          ^ (Value.fmt value) ^ "</attr>\n" 

      fun fmt' (attr, acc) = (fmtAttr attr) ^ acc
    in
      "<entry name=\"" ^ name ^ "\">" ^ (Dict.foldl fmt' "</entry>\n" e)
    end
    | fmt (name, NIL' s) = "<nil name=\"" ^ name ^ "\"></nil>\n"

  (* saves an entry, given its name *)
  (* val save : TextIO.outstream -> (string * entry') -> unit *)
  fun save sout (name, entry) =
    TextIO.output (sout, fmt (name, entry))

  fun getname (e) =
    case Dict.find (e, "entry") of
       (SOME ({value=(Value.Single name),...} : attribute')) => name
      | _ => "?"

  (* needs to enforce validation:
   subdataset *)
  fun validate (_, "entry", _) = raise RenameNotImpemented
    | validate (e, "modtime", SOME _) = raise EnforcedValidation (getname e, 
								  "modtime")
    | validate _ = ()
   
  (* val storedata : (ident * (string -> acl)) * (storedata list * entry) *)
  fun storedata (ident, aclf) =
    let
      val myrights = Acl.myrights ident
      fun checkAcl rights acl = 
	if AclRights.has (myrights acl, rights)
	  then () 
	else (print ("denying: " ^ (Acl.toString acl) ^ "\n");
	      print ("   lacks " ^ (AclRights.toString rights));
	      raise NotPermitted)
      val checkInsert = checkAcl AclRights.ACLi
      val checkWrite = checkAcl AclRights.ACLw
      val checkAdmin = checkAcl AclRights.ACLa

      fun storedata' (nil, map) = map
	| storedata' ((item as {acl, attribute, 
				value=(value as SOME Value.Default)})::sd, 
		      map) =
	(validate (map, attribute, value);
	 storedata' (sd, (#1 (Dict.remove (map, attribute))
			  handle LibBase.NotFound => map)))
	| storedata' ((item as {acl, attribute, value})::sd, map) =
	(validate (map, attribute, value);
	 case Dict.find (map, attribute) of
	   (NONE) => 
	     let
	       val _ = checkInsert (aclf attribute)

	       val value = case value of
		 NONE => Value.Default
	       | (SOME v) => v
		   
	       val vsize = Value.size value
	     in
	       storedata' (sd, Dict.insert (map, attribute,
					    {attribute=attribute,
					     acl=acl,
					     size=vsize,
					     value=value}))
	     end
	 | (SOME {acl=origAcl, attribute=_, size=_, value=origValue}) =>
	     let
	       val acl = if isSome acl 
			   then (checkAdmin (aclf attribute);
				 acl)
			 else origAcl
			   
	       val value = if isSome value 
			     then (checkWrite (getOpt(origAcl, aclf attribute));
				   valOf value)
			   else origValue
	     in
	       storedata' (sd, Dict.insert (map, attribute,
					    {acl=acl, attribute=attribute,
					     size=Value.size value,
					     value=value}))
	     end)
    in
      storedata'
    end

  (* creates an entry, given it's name *)
  (* val create : Acl.auth.ident -> (string -> acl) -> string -> 
                    storedata list -> entry *)
  fun create ident aclf shortname sd =
    let
      (* must have the following attributes:
       "entry", "modtime" (may not appear in create) *)
      val entryattr = ({acl=NONE,attribute="entry",
			size=[String.size shortname],
			value=(Value.Single shortname)} : attribute')

      val now = AcapTime.toString (AcapTime.now ())
      val modtimeattr = ({acl=NONE, attribute="modtime",
			  size=[String.size now], value=(Value.Single now)} 
			        : attribute')

      val map = Dict.insert (Dict.empty, "entry", entryattr)
      val map = Dict.insert (map, "modtime", modtimeattr)

      (* convert the rest of the storedata to attributes *)
      val map = storedata (ident, aclf) (sd, map)
    in
      E' map
    end

  (* stores a list of attributes in this entry; this is where
     enforced validation takes place. returns NONE if this entry
     was deleted *)
  (* requires current user and an default attrname -> acl function
       raises: EnforcedValidation, NotPermitted, Modified *)
  (* val store : Acl.auth.ident -> (string -> Acl.acl) -> entry -> 
    (storedata list * AcapTime.acaptime option) -> entry option *)
  fun store ident aclf (E' map) ([{attribute="entry", acl=NONE,
				     value=(SOME Value.Default)}], time) =
                 (* trying to delete the entry *)
    let
      (* check the time *)
      val _ = case time of
	NONE => ()
      | (SOME t) => 
	  let
	    val thn = case Dict.find (map, "modtime") of
	      NONE => raise (InternalError "entry with no modtime")
	    | (SOME {value=Value.Single str,...}) => AcapTime.fromString str
	    | _ => raise (InternalError "entry with badly formed modtime")
	  in
	    if AcapTime.compare (t, thn) = LESS then raise (Modified thn)
	    else ()
	  end
    in
      NONE
    end
    | store ident aclf (NIL' _) ([{attribute="entry", acl=NONE,
				   value=(SOME Value.Default)}], time) = NONE
    | store ident aclf (E' map) ([{attribute="entry", acl=NONE,
				   value=(SOME Value.Nil)}], time) =
                 (* trying to delete the entry & replay it by a NIL *)
    let
      val _ = case time of
	NONE => ()
      | (SOME t) => 
	  let
	    val thn = case Dict.find (map, "modtime") of
	      NONE => raise (InternalError "entry with no modtime")
	    | (SOME {value=Value.Single str,...}) => AcapTime.fromString str
	    | _ => raise (InternalError "entry with badly formed modtime")
	  in
	    if AcapTime.compare (t, thn) = LESS then raise (Modified thn)
	    else ()
	  end

      val s = case Dict.find (map, "entry") of
	NONE => raise (InternalError "entry with no entry name")
      | (SOME {value=s,...})  => s
    in
      SOME (NIL' s)
    end
    | store ident aclf (E' map) (sd, time) =
    let
      (* check the time *)
      val _ = case time of
	NONE => ()
      | (SOME t) => 
	  let
	    val thn = case Dict.find (map, "modtime") of
	      NONE => raise (InternalError "entry with no modtime")
	    | (SOME {value=Value.Single str,...}) => AcapTime.fromString str
	    | _ => raise (InternalError "entry with badly formed modtime")
	  in
	    if AcapTime.compare (t, thn) = LESS then raise (Modified thn)
	    else ()
	  end

      val now = AcapTime.toString (AcapTime.now ())
      val modtimeattr = ({acl=NONE, attribute="modtime",
			  size=[String.size now], value=(Value.Single now)} 
			        : attribute')

      val map = Dict.insert (map, "modtime", modtimeattr)
      val map = storedata (ident, aclf) (sd, map)
    in
      SOME (E' map)
    end
    | store ident aclf (NIL' (Value.Single s)) (sd, time) = 
    SOME (create ident aclf s sd)
    | store _ _ _ _ = raise (InternalError "badly formed value in NIL' entry")

  (* gets the attributes *)
  (* val fetch : Acl.auth.ident -> (string -> Acl.acl) -> entry -> 
    string -> attribute *)
  fun fetch ident (E map) a =
    let
      val myrights = Acl.myrights ident
      fun checkAcl rights acl = AclRights.has (myrights acl, rights)
      val checkRead = checkAcl AclRights.ACLr
    in
      case Dict.find (map, a) of
	NONE => (makeNull (a, Acl.empty))
      | (SOME (attr as {acl=acl,...})) => 
	  if checkRead acl then attr
	  else (makeNull (a, acl))
    end
    | fetch ident (NIL s) a = makeNull (a, Acl.empty)

  (* val getattr : entry -> string -> attribute' option *)
  fun getattr' (E' e) a = Dict.find (e, a)
    | getattr' (NIL' s) "entry" = 
    SOME {attribute="entry", acl=NONE, value=s, size=Value.size s}
    | getattr' _ _ = NONE

  (* gets the matching attributes *)
  (*  val search : Acl.auth.ident -> (string -> Acl.acl) -> entry -> 
    (string -> bool) -> attribute list *)
  fun search ident (E map) search =
    let
      val myrights = Acl.myrights ident
      fun checkAcl rights acl = AclRights.has (myrights acl, rights)
      val checkRead = checkAcl AclRights.ACLr

      fun dosearch (attr as {attribute,acl=acl,...} : attribute, acc) =
        if search attribute andalso checkRead acl then
	  attr::acc
	else acc
    in
      Dict.foldl dosearch nil map
    end
    | search ident (NIL s) search = nil

  (* return the time of last modification *)
  (* val gettime : entry -> AcapTime.acaptime *)
  fun gettime (E entry) =
    let
      val t = case Dict.find (entry, "modtime") of
	NONE => raise (InternalError "entry with no modtime")
      | (SOME {value=Value.Single str,...}) => AcapTime.fromString str
      | _ => raise (InternalError "entry with badly formed modtime")
    in
      t
    end
    | gettime _ = AcapTime.old

  (* val union : (entry * entry) -> entry
   if an entry exists, we want to take the child;
     possible ramifications for NIL entries/attributes here *)
  fun union_attr ("modtime", 
		  a1 as {acl=_, attribute=_, 
			 size=xsz, value=vx as (Value.Single x)},
		  a2 as {acl, attribute,
			 size=ysz, value=vy as (Value.Single y)}) = 
    (* take the latest *)
    (case AcapTime.compare (AcapTime.fromString x, AcapTime.fromString y) of
       LESS => {acl=acl, attribute=attribute, size=xsz, value=vx}
     | (EQUAL | GREATER) => a2)
    | union_attr (_, a1, a2) = (* take the child *) a2

  fun union (E e1, E e2) = E (Dict.unionWithi union_attr (e1, e2))
    | union (_, e2) = e2 (* otherwise, just take the child *)

  (* val fillAcls : (string -> acl) -> entry' -> entry *)
  fun fillAcls aclf (E' map) =
    let
      fun fill ({acl=NONE, attribute, size, value} : attribute') =
	{acl=(aclf attribute), attribute=attribute, size=size, value=value}
	| fill ({acl=SOME acl, attribute, size, value}) =
	{acl=acl, attribute=attribute, size=size, value=value}
    in
      E (Dict.map fill map)
    end
    | fillAcls _ (NIL' s) = (NIL s)

  (* used for transferring information (dataset->dataset, dataset->context) *)
  datatype updatemsg =
    NEW of {dset : string, name : string, entry : entry}
  | CHANGE of {dset : string, name : string, entry : entry}
  | DELETE of {dset : string, name : string}

end
