(* 
 * Copyright (c) 2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *)

(* ACAP entries 

   Generally, entries are the thing that are passed around by the
   various routines.
*)

signature ENTRY =
sig
  type entry
  type entry' (* entry without filled-in acls *)
  type attribute = {acl : Acl.acl, attribute : string,
		    size : int list, value : Value.value}
  type attribute' = {acl : Acl.acl option, attribute : string,
		     size : int list, value : Value.value}
  type storedata = {acl : Acl.acl option, attribute : string, 
		    value : Value.value option}

  (* used for transferring information (dataset->dataset, dataset->context) *)
  datatype updatemsg =
    NEW of {dset : string, name : string, entry : entry}
  | CHANGE of {dset : string, name : string, entry : entry}
  | DELETE of {dset : string, name : string}

  (* raised when an attempt to store an illegal value is made to this
   dataset---returns attribute raised on *)
  exception EnforcedValidation of (string * string)

  (* raised when an attempt is made to store with a UNCHANGEDSINCE;
     returns the time it was last modified *)
  exception Modified of AcapTime.acaptime

  (* renames aren't implemented *)
  exception RenameNotImpemented

  (* ACL problem *)
  exception NotPermitted
   
  (* read an entry *)
  val scan : (char, 'strm) StringCvt.reader 
    -> (string * entry', 'strm) StringCvt.reader

  (* saves an entry, given it's name *)
  val save : TextIO.outstream -> (string * entry') -> unit

  (* creates an entry, given it's complete name *)
  val create : Auth.ident -> (string -> Acl.acl) -> 
    string -> storedata list -> entry'
  
  (* stores a list of attributes in this entry; this is where
     enforced validation takes place. returns NONE if this entry
     was deleted *)
  (* requires current user and an default attrname -> acl function *)
  val store : Auth.ident -> (string -> Acl.acl) -> entry' -> 
    (storedata list * AcapTime.acaptime option) -> entry' option

  (* gets the attributes *)
  val fetch : Auth.ident -> entry -> string -> attribute

  (* gets the matching attributes *)
  val search : Auth.ident -> entry -> (string -> bool) -> attribute list

  (* return the time of last modification *)
  val gettime : entry -> AcapTime.acaptime

  (* --- needed for inheritance --- *)

  (* given an (ancestor, entry) pair, create an inherited entry *)
  val union : (entry * entry) -> entry
    
  (* given an entry and an acl function, produce a fully qualified entry *)
  val fillAcls : (string -> Acl.acl) -> entry' -> entry

  val getattr' : entry' -> string -> attribute' option
end
