(* ACAP throttles.

   A throttle regulates how much data can be sent through.  There are
   two types, a hard throttle and a soft throttle.

   Hard throttles queue up data making sure it is less then the
   specified number.  If more occur it raises WAYTOOMANY.

   Soft throttles queue up data making sure that if more than the
   specified number are returned, only the first so many go through,
   and raising TOOMANY. 

   The user of the throttles must know what to do with the return value.

   Note that throttles inherently have state; the implementation may
   NOT be thread safe, so beware of calling from multiple threads!

   *)

signature THROTTLE =
sig
  type 'a throttle

  datatype hardresult = H_WAYTOOMANY | H_OK
  datatype softresult = S_TOOMANY of int | S_OK
  datatype result = OK

  (* create a hard throttle, plus what to call when done *)
  val createhard : int -> ('a -> unit) -> (hardresult -> unit) -> 'a throttle

  (* create a soft throttle; the first number is the limit *)
  val createsoft : (int * int) -> ('a -> unit) -> (softresult -> unit) -> 
                   'a throttle

  (* create a sorting throttle *)
  val createsort : (('a * 'a) -> order) -> ('a -> unit) -> 
                   (result -> unit) -> 'a throttle

  (* a throttle that just passes through data *)
  val createpass : ('a -> unit) -> (result -> unit) -> 'a throttle

  (* send data into the throttle *)
  val call : 'a throttle -> 'a -> unit

  (* indicates that no more data will be coming through the throttle;
     if the limit is exceeded it just throws out the data *)
  val done : 'a throttle -> unit

end
