structure Dataset :> DATASET =
struct
  structure T = Throttle

  (* shortcuts! *)
  structure SV = LSyncVar
  structure MC = RMulticast
  structure V = Value

  structure Dict = RedBlackMapFn(type ord_key = String.string
				 val compare = String.compare)

  (* structure EL = ListMapFn(type ord_key = String.string
			   val compare = String.compare)*)
  structure EL = RedBlackMapFn(type ord_key = String.string
			       val compare = String.compare)

  (* exported entries must have all ACLs fully filled in *)
  datatype dataset =
    OnDisk of {channel : Entry.updatemsg MC.mchan,
	       dieListener : unit SV.ivar option}
  | InMem of {data : Entry.entry' EL.map, idata : Entry.entry EL.map,
	      channel : Entry.updatemsg MC.mchan,
	      dieListener : unit SV.ivar option,
	      inherits : string option, 
	      owner : Auth.ident,
	      acl : string -> Acl.acl,
	      flag : bool}
  | Remote of string list

  datatype DatasetDict = EMPTY
                       | DD of ((dataset SV.mvar) Dict.map) SV.mvar

  exception NoSuchDataset
  exception DatasetRefer of string list
  exception Impossible of string
  exception SyntaxError of string
  exception DiskError of string

  val dict = ref EMPTY

  (* random number server *)
  val randCh = (CML.channel ()) : int CML.chan

  (* functions to manipulate the dataset file *)
  fun writeDSets mydict = 
    let
      val fmtRem = List.foldl (fn (url, acc) => url ^ "\t" ^ acc) ""

      fun fmt (OnDisk _) = ("\tlocal\n")
	| fmt (InMem _) = ("\tlocal\n")
	| fmt (Remote urls) = ("\tremote\t" ^ (fmtRem urls) ^ "\n")

      val fnamenew = Config.FNAME_dsets ^ ".NEW"
      val f = TextIO.openOut fnamenew

      fun out (key, data) = TextIO.output (f, key ^ (fmt (SV.mGet data)))
    in
      Dict.appi out mydict;
      TextIO.closeOut f;
      Posix.FileSys.rename {old=fnamenew, new=Config.FNAME_dsets}
    end

  fun readDSets () = 
   (let
      (* blank out dict *)
      val _ = dict := EMPTY

      val scanName = ParserComb.token (fn c => c <> #"\t")

      fun newlocal _ = OnDisk ({channel=MC.mChannel (),
				dieListener=NONE})
      val scanLocal = ParserComb.skipBefore (fn c => c = #"\t") 
	              (ParserComb.wrap (ParserComb.string "local", newlocal))

      fun newremote (_, remstr) = (Remote remstr)
      val url = ParserComb.seqWith (#1) (ParserComb.token (fn c => c <> #"\t"),
					 ParserComb.char #"\t")
      val scanRemote = 
	ParserComb.seqWith newremote (ParserComb.string "remote\t",
			 ParserComb.zeroOrMore (url))

      val scanDSet = ParserComb.wrap (ParserComb.or (scanLocal, scanRemote),
				      SV.mVarInit)

      val scanEntry = ParserComb.seqWith (#1)
	(ParserComb.seq (scanName, scanDSet),
	 ParserComb.char #"\n")

      fun makedict getc =
	let 
	  val scanEntry = scanEntry getc
	  fun makedict' (sin, dict) =
	    case scanEntry sin of
	      NONE => dict
	    | (SOME (pair, sin)) => makedict' (sin, Dict.insert' (pair, dict))
	in
	  makedict'
	end

      val f = TextIO.openIn Config.FNAME_dsets
      val sin = TextIO.getInstream f
      val d = makedict TextIO.StreamIO.input1 (sin, Dict.empty)
    in
      dict := (DD(SV.mVarInit d));
      TextIO.closeIn f;
      true
    end
  handle _ => false)

  datatype scanning =
      gotE of (string * Entry.entry') 
    | delE of (string * AcapTime.acaptime)
    | DONE

  fun readSet (dsetname) =
    let
      val scanOpen = ParserComb.string "<delete name=\""
      val scanName = ParserComb.token (fn c => c <> #"\"")
      val scanTime = ParserComb.seqWith (AcapTime.fromString o #2)
		      (ParserComb.string "\" time=\"",
		       ParserComb.token (fn c => c <> #"\""))
      val scanClose = ParserComb.string "\"></delete>"

      val scanDelete = ParserComb.seqWith #2 (scanOpen, scanName)
      val scanDelete = ParserComb.seq (scanDelete,
				       ParserComb.seqWith #1 (scanTime,
							      scanClose))
      val scanDelete = ParserComb.skipBefore Char.isSpace scanDelete
      val scanDelete = ParserComb.wrap (scanDelete, delE)
      val scanEntry = ParserComb.skipBefore Char.isSpace Entry.scan
      val scanEntry = ParserComb.wrap (scanEntry, gotE)
      val scanNone = ParserComb.result (DONE)

      val scanOne = ParserComb.or' [scanEntry, scanDelete, scanNone]

      fun doRemove (data, n) = 
        (let 
	   val (data, _) = EL.remove (data, n)
	 in
	   data
	 end handle LibBase.NotFound => data)

      fun scanData data DONE = ParserComb.result data
	| scanData data (gotE (e as (n, _))) =
	(ParserComb.bind (scanOne, scanData (EL.insert' (e, data))))
	| scanData data (delE (n, time)) =
	(ParserComb.bind (scanOne, scanData (doRemove (data, n))))

      val scan = ParserComb.bind (scanOne,
				  scanData EL.empty)

      val file = Config.makeDataFile dsetname
      val f = TextIO.openIn file

      val data = 
	case TextIO.scanStream scan f of
	  NONE => raise (DiskError "badly formed dataset")
	| (SOME d) => d

      val _ = TextIO.closeIn f
    in
      data
    end

  datatype change = 
    addE of (string * Entry.entry')
  | delE of (string * AcapTime.acaptime)
  (* val writeSet : (string * entry' EL.map * change) *)
  fun writeSet (dsetname, data, change) =
    let
      val file = Config.makeDataFile dsetname

      fun rewriteFile () =
	let
	  val _ = TextIO.print ("compacting " ^ dsetname ^ "...\n")
	  val newfile = file ^ ".NEW"
	  val strm = TextIO.openOut newfile
	  val _ = EL.appi (Entry.save strm) data
	in
	  TextIO.closeOut strm;
	  Posix.FileSys.rename {old=newfile, new=file}
	end

      fun saveDelete sout (name, time) =
	TextIO.output (sout, 
          "<delete name=\"" ^ name ^ "\" time=\"" ^ (AcapTime.toString time)
		            ^ "\"></delete>\n")

      fun appendEntry () =
	let
	  val strm = TextIO.openAppend file
	  val _ = 
	    case change of 
	      (addE pair) => Entry.save strm pair
	    | (delE pair) => saveDelete strm pair
	in
	  TextIO.closeOut strm
	end

      val setsize = EL.numItems data

      val rand = (CML.recv randCh) mod setsize
    in
      if rand = 0 then (* write the whole thing *)
	rewriteFile ()
      else (* just append this entry *)
	appendEntry ()
    end

  val defaultAclf = (fn s => Acl.empty)

  (* create a new root dataset *)
  fun createRoot () =
    let
      val acl = Value.List ["$anyone\txrwia"]

      val null = Entry.create Auth.master defaultAclf
	"" [{acl=NONE, attribute="dataset.owner", 
	      value=SOME (Value.Single (Auth.toString Auth.master))},
	     {acl=NONE, attribute="dataset.acl",
	      value=SOME acl}]

      val acl = Acl.fromValue acl
      fun aclf s = acl

      val data = EL.singleton ("", null)
	
      (* commit it to disk *)
      val _ = writeSet ("/", data, addE("", null))

      val root = InMem ({data=data, 
			 idata=EL.map (Entry.fillAcls aclf) data,
			 channel=MC.mChannel (),
			 dieListener=NONE,
			 inherits=NONE,
			 owner=Auth.master,
			 acl=aclf,
			 flag=true})

      val mydict = Dict.insert (Dict.empty, "/", SV.mVarInit root)
    in
      (* initialize the in memory dictionary *)
      writeDSets mydict;
      dict := DD(SV.mVarInit mydict)
    end
  
  (* sweeper *)
  fun sweeper (root, force) =
    let
      fun examine (name, d) =
	let
          (* no point in looking at it if it's currently locked *)
	  val d' = SV.mTakePoll d
	in
	 case d' of
	   NONE => ()
	 | (SOME d') =>
	   let
	     val d' = 
	       case d' of 
		 (InMem {data, idata,
			 channel,
			 dieListener,
			 inherits,
			 owner,
			 acl,
			 flag=true}) => (InMem {data=data, idata=idata,
						channel=channel,
						dieListener=dieListener,
						inherits=inherits,
						owner=owner,
						acl=acl,
						flag=false})
	       | (InMem {data, idata,
			 channel,
			 dieListener,
			 inherits,
			 owner,
			 acl,
			 flag=false}) => 
		 if MC.listeners channel = 0 then
		   (print ("   flushing " ^ name ^ "\n");
		    (case dieListener of
		       NONE => ()
		     | (SOME die) => SV.iPut (die, ()));
		       OnDisk {channel=channel,
			       dieListener=NONE})
		 else OnDisk {channel=channel, dieListener=dieListener}
		     | _ => d'
	   in
	     SV.mPut (d, d')
	   end
	end

      val forceEvt = CML.recvEvt (force)
      val timeEvt = CML.timeOutEvt (Config.SweepTime)
    in
      (* wait for timeout or a force *)
      CML.select [forceEvt, timeEvt];

      (* do sweep *)
      TextIO.print "sweeping...\n";
      Dict.appi examine (SV.mGet root);
      TextIO.print "done.\n";

      (* loop forever *)
      sweeper (root, force)
    end

  fun randServer () =
    let
      val rand = 
	Random.rand (134, Int32.toInt (Time.toSeconds (Time.now ()) div 2))

      fun loop () =
	(CML.send (randCh, Random.randInt rand);
	 loop ())
    in
      ignore (CML.spawn loop)
    end

  (* initialize the dictionary; read off of disk *)
  fun init () = 
    (randServer ();
     if readDSets () then ()
     else createRoot ();
     case dict of (ref EMPTY) => ()
                | (ref (DD mvar)) => ignore (CML.spawnc sweeper 
					     (mvar, CML.channel ())))

  fun update (dset, oldEntry, newEntry) =
    let
      fun insert' (channel, data, idata, aclf, name, ientry) =
	let
	  val entry = EL.find (data, name)
	    
	  val newientry = case entry of
	    NONE => ientry
	  | (SOME entry) => Entry.union (ientry, Entry.fillAcls aclf entry)
	      
	in
	  case entry of
	    NONE => (MC.multicast (channel, Entry.NEW(dset ^ name, newientry));
		     EL.insert (idata, name, ientry))
	  | (SOME _) => (MC.multicast (channel, 
				       Entry.CHANGE((dset ^ name, ientry),
						    (dset ^ name, newientry)));
			 EL.insert (idata, name, ientry))
	end

      fun change' (channel, data, idata, aclf, name, oldientry, newientry) =
	let
	  val entry = EL.find (data, name)
	    
	  val oldientry = valOf (EL.find (idata, name))

	  val ientry = case entry of
	    NONE => newientry
	  | (SOME e) => Entry.union (newientry, Entry.fillAcls aclf e)
	in
	  MC.multicast (channel, Entry.CHANGE((dset ^ name, oldientry),
					      (dset ^ name, ientry)));
	  EL.insert (idata, name, ientry)
	end

      fun delete' (channel, data, idata, aclf, name) =
	let
	  val (idata, oldientry) = EL.remove (idata, name)
 
	  val entry = EL.find (data, name)
	  val ientry = case entry of
	    NONE => NONE
	  | (SOME entry) => SOME (Entry.fillAcls aclf entry)
	in
	  case ientry of 
	    NONE => (MC.multicast (channel, Entry.DELETE(dset ^ name, 
							 oldientry));
		     idata)
	  | (SOME ientry) => 
	      (MC.multicast (channel, Entry.CHANGE ((dset ^ name, oldientry),
						    (dset ^ name, ientry)));
	       EL.insert (idata, name, ientry))
	end

      fun shorten n = Substring.string 
	(Substring.taker (fn c => c <> #"/") (Substring.all n))

      val mydict = 
	case dict of (ref EMPTY) => raise (Impossible "no dset dictionary")
                   | (ref (DD mvar)) => SV.mGet mvar

      val d = case Dict.find (mydict, dset) of
	NONE => raise (Impossible "thread listening for non-existant dset")
      | (SOME d) => d

      val d' = SV.mTake d
      val d' = load (dset, d')
      val d' = doInheritance (dset, mydict, d')

      (* invariant: d' is now an InMem structure ! *)
      val ({data, idata,
	    channel,
	    dieListener,
	    inherits,
	    owner,
	    acl,
	    flag}) = case d' of
	               (InMem stuff) => stuff
		     | _ => raise (Impossible "dataset not loaded")

      (* deletion? *)
      val idata = 
	case (oldEntry, newEntry) of
	  ((SOME (n, e1)), NONE) => delete' (channel, data, idata, acl, 
					     shorten n)
	| (SOME (oldn, e1), SOME (newn, _)) => 
	    if oldn <> newn then delete' (channel, data, idata, acl, 
					  shorten oldn)
	    else idata
	| _ => idata
	    
      (* insertion? *)
      val idata = 
	case (oldEntry, newEntry) of
	  (NONE, (SOME (n, e2))) => insert' (channel, data, idata, acl,
					     shorten n, e2)
	| (SOME (oldn, _), SOME (newn, e2)) => 
	    if oldn <> newn then insert' (channel, data, idata, acl,
					  shorten newn, e2)
	    else idata
	| _ => idata
	    
      (* change? *)
      val idata = 
	case (oldEntry, newEntry) of
	  (SOME (oldn, e1), SOME (newn, e2)) => 
	    if oldn = newn then change' (channel, data, idata, acl,
					 shorten newn, e1, e2)
	    else idata
	| _ => idata
			 
      val d' = InMem {data=data, idata=idata,
		      channel=channel,
		      dieListener=dieListener,
		      inherits=inherits,
		      owner=owner,
		      acl=acl,
		      flag=flag}
    in
      SV.mPut (d, d')
    end

  (* listener stuff *)
  and listener (args as (dset, evts, port)) =
    case CML.select (evts) of
      (SOME (Entry.NEW new)) => (update (dset, NONE, SOME new); listener args)
    | (SOME (Entry.CHANGE (a,b))) => (update (dset, SOME a, SOME b); 
				      listener args)
    | (SOME (Entry.DELETE old)) => (update (dset, SOME old, NONE); 
				    listener args)
    | NONE => (MC.release port)

  and startListener (dset, die, port) =
    let
      val dieevt = CML.wrap (SV.iGetEvt (die), fn () => NONE)
      val evt = CML.wrap (MC.recvEvt port, SOME)
    in
      ignore (CML.spawnc listener (dset, [dieevt, evt], port))
    end

  (* load dset

   does NOT initialize inheritance *)
  and load (name, dset as (Remote _)) = dset
    | load (name, dset as (InMem {data, idata,
				  channel,
				  dieListener,
				  inherits,
				  owner,
				  acl,
				  flag})) = InMem {data=data, idata=idata,
						   channel=channel,
						   dieListener=dieListener,
						   inherits=inherits,
						   owner=owner,
						   acl=acl,
						   flag=true}
    | load (name, dset as (OnDisk {channel, dieListener})) =
    let
      (* load all the entries *)
      val data = readSet name
      val null = 
	case EL.find (data, "") of
	  NONE => raise (DiskError "no null entry for dataset")
	| (SOME null) => null

      val owner =
	case Entry.getattr' null "dataset.owner" of
	  NONE => raise (DiskError "null entry doesn't have dataset.owner")
	| (SOME {value=V.Single owner,...}) => Auth.fromString owner
	| _ => raise (DiskError "dataset.owner is malformed")

      val inherits = 
	case Entry.getattr' null "dataset.inherit" of
	  NONE => NONE
	| (SOME {value=V.Single inherit,...}) => SOME inherit
	| (SOME {value=V.Nil,...}) => NONE
	| _ => raise (DiskError "dataset.inherit is malformed")

      fun acl attrname =
	case Entry.getattr' null ("dataset.acl." ^ attrname) of
	  NONE => (case Entry.getattr' null "dataset.acl" of
		     NONE => Acl.empty
		   | (SOME {value=aclval,...}) => Acl.fromValue aclval)
	| (SOME {value=aclval,...}) => Acl.fromValue aclval

      val _ = print ("Done loading " ^ name ^ "\n")
    in
      InMem {data=data, idata=EL.empty,
	     channel=channel,
	     dieListener=dieListener,
	     inherits=inherits,
	     owner=owner,
	     acl=acl,
	     flag=true}
    end

  (* create a dataset given it's path & a default "" entry;
   the parent dataset must exist

   it does NOT initialize inheritance *)
  and create (path, last, {null, owner, inherits, aclf}) =
    let 
      val dir = Config.FDIR_spool ^ path ^ last
	
      val _ = print ("making " ^ dir ^ "\n")

      val flags = Posix.FileSys.S.flags [Posix.FileSys.S.irwxu,
					 Posix.FileSys.S.irgrp,
					 Posix.FileSys.S.ixgrp,
					 Posix.FileSys.S.iroth,
					 Posix.FileSys.S.ixoth]

      (* create the directory *)
      val _ = Posix.FileSys.mkdir (dir, flags)

      val inherits = 
	case inherits of
	  NONE => NONE
	| (SOME v) => SOME (v ^ last ^ "/")

      val null = 
	case inherits of
	  (SOME v) => 
	    valOf (Entry.store Auth.master defaultAclf null
		   ([{acl=NONE,attribute="dataset.inherit",
		      value=SOME (V.Single v)}], 
		    NONE))
	| NONE => null

      (* whoever created this is the owner *)
      val null = valOf (Entry.store Auth.master defaultAclf null
			([{acl=NONE,attribute="dataset.owner",
			  value=SOME (V.Single (Auth.toString owner))}],
			 NONE))

      val data = EL.singleton ("", null)
      val name = path ^ last ^ "/"
      val _ = writeSet (name, data, addE("", null))
      val channel = MC.mChannel ()

      (* announce insertion of null entry *)
      val _ = MC.multicast (channel, Entry.NEW(name, Entry.fillAcls aclf null))
	
      val dset = InMem {data=data, idata=EL.empty,
			channel=channel,
			dieListener=NONE,
			inherits=inherits,
			owner=owner,
			acl=aclf,
			flag=true}
    in
      dset
    end

  (* fetch a dset from the dictionary, loading it if needed,
   can raise NoSuchDataset *)
  and fetch (mydict, dset) =
    case Dict.find (mydict, dset) of
      NONE => NONE
    | (SOME d) => 
	let
	  val d' = SV.mTake d
	  val d' = load (dset, d')
	  val d' = doInheritance (dset, mydict, d')
	in
	  SV.mPut (d, d');
	  SOME d'
	end

  (* given a dataset, do the inheritance if not done already;
   this is where a thread should be started for listening if needed *)
  and doInheritance (dsetname, _, dset as InMem {data, idata,
						 channel,
						 dieListener,
						 inherits=NONE,
						 owner,
						 acl,
						 flag}) =
    if EL.isEmpty idata then
      InMem {data=data, idata=EL.map (Entry.fillAcls acl) data,
	     channel=channel,
	     dieListener=dieListener,
	     inherits=NONE,
	     owner=owner,
	     acl=acl,
	     flag=flag}
    else dset
    | doInheritance (dsetname, mydict, dset as InMem {data, idata,
						      channel,
						      dieListener,
						      inherits=SOME ancestor,
						      owner,
						      acl,
						      flag}) =
    if EL.isEmpty idata then
      (* ok, grab what we inherit from *)
      case fetch (mydict, ancestor) of (* it doesn't actually exist *)
	NONE => InMem {data=data, idata=EL.map (Entry.fillAcls acl) data,
		       channel=channel,
		       dieListener=dieListener,
		       inherits=SOME ancestor,
		       owner=owner,
		       acl=acl,
		       flag=flag}
      | (SOME (InMem {idata=adata,channel=achannel,...})) => 
	  let
	    val fdata = EL.map (Entry.fillAcls acl) data
	    val idata = EL.unionWith Entry.union (adata, fdata)
	    val die = SV.iVar ()
	    val _ = startListener (dsetname, die, MC.port achannel)
	    val dieListener = SOME die
	  in
	    InMem {data=data, idata=idata,
		   channel=channel,
		   dieListener=dieListener,
		   inherits=SOME ancestor,
		   owner=owner,
		   acl=acl,
		   flag=flag}
	  end
      | _ => 
	  raise (Impossible "trying to compute inheritance from on-disk data")
    else dset (* inheritance already done *)
    | doInheritance (_, _, dset) = dset

  and doInheritanceEntry (_, dset as InMem {data, idata,
					    channel,
					    dieListener,
					    inherits=NONE,
					    owner,
					    acl,
					    flag}, (name, SOME entry)) =
    InMem {data=data, idata=EL.insert (idata, name, Entry.fillAcls acl entry),
	   channel=channel,
	   dieListener=dieListener,
	   inherits=NONE,
	   owner=owner,
	   acl=acl,
	   flag=flag}
    | doInheritanceEntry (mydict, dset as InMem {data, idata,
						 channel,
						 dieListener,
						 inherits=SOME ancestor,
						 owner,
						 acl,
						 flag}, (name, SOME entry)) =
      (* ok, grab what we inherit from *)
    (case fetch (mydict, ancestor) of (* it doesn't actually exist *)
       NONE => InMem {data=data, idata=EL.insert (idata, name, 
						  Entry.fillAcls acl entry),
		      channel=channel,
		      dieListener=dieListener,
		      inherits=SOME ancestor,
		      owner=owner,
		      acl=acl,
		       flag=flag}
     | (SOME (InMem {idata=adata,...})) => 
	 let
	   val entry = Entry.fillAcls acl entry
	   val ientry = EL.find (adata, name)
	   val idata = case ientry of
	     NONE => EL.insert (idata, name, entry)
	   | (SOME ientry) => EL.insert (idata, name, Entry.union (ientry, entry))
	 in
	   InMem {data=data, idata=idata,
		   channel=channel,
		   dieListener=dieListener,
		   inherits=SOME ancestor,
		   owner=owner,
		   acl=acl,
		   flag=flag}
	  end
     | _ => raise (Impossible "trying to compute inheritance from on-disk data"))
    | doInheritanceEntry (mydict, dset as InMem {data, idata,
						 channel,
						 dieListener,
						 inherits,
						 owner,
						 acl,
						 flag}, (name, NONE)) =
      (case (case inherits of NONE => NONE 
                            | (SOME ancestor) => fetch (mydict, ancestor)) of
	 NONE => (* no inheritance *)
	   InMem {data=data, idata=(#1 (EL.remove (idata, name)))
		                  handle LibBase.NotFound => idata,
		  channel=channel,
		  dieListener=dieListener,
		  inherits=inherits,
		  owner=owner,
		  acl=acl,
		  flag=flag}
       | (SOME (InMem {idata=adata,...})) => 
	   InMem {data=data, idata=case EL.find (adata, name) of
		                      NONE => (#1 (EL.remove (idata, name))
					            handle LibBase.NotFound => idata)
				    | (SOME a) => EL.insert (idata, name, a),
		  channel=channel,
		  dieListener=dieListener,
		  inherits=inherits,
		  owner=owner,
		  acl=acl,
		  flag=flag}
       | _ => raise (Impossible "trying to compute inheritance from on-disk data"))
    | doInheritanceEntry (_, dset, _) = dset
  
  (* find and load the datasets starting with prefix, to depth *)
  fun findDSets (n, prefix, mydict, messenger) = 
    let
      val countSlash = 
	(Substring.foldl (fn (c, a) => if c = #"/" then a+1 else a) 0)
      o (Substring.all)

      val allowed = n + (countSlash prefix)
      val sub = String.isPrefix prefix
      val good = if n = 0 then sub
		 else fn name => sub name andalso countSlash name < allowed

      fun f (name, d) =
	if good name then
	  let
	    val d' = SV.mTake d
	    val d' = load (name, d')
	    val d' = doInheritance (name, mydict, d')

	    val (InMem {channel,...}) = d'

	    val _ = (* set up notifications if needed *)
	      case messenger of
		NONE => ()
	      | (SOME messenger) => messenger (name, MC.port channel)
	  in
	    SV.mPut (d, d');
	    SOME d'
	  end
	else NONE
    in
      Dict.mapPartiali f mydict
    end

  fun search ident (dset, depth, doinherit) 
                   sfun
                   messenger 
                   (out, referral) 
                   finish = 
    let
      fun searchOne (name, Remote urls) = referral (name, urls)
	| searchOne (name, InMem {data, idata, 
				  channel,
				  acl, inherits, ...}) =
	let
	  fun win (n, e) = if sfun e then out (name, n, e) else ()

	  val fill = Entry.fillAcls acl
	  fun win' (n, e') = 
	    let val e = fill e'
	    in if sfun e then out (name, n, e) else () end
	in
	  if doinherit orelse not (isSome inherits) then EL.appi win idata
	  else EL.appi win' data
	end
	| searchOne _ = 
	raise (Impossible "trying to search dataset not in memory")

      val mydict =
	case dict of (ref EMPTY) => raise (Impossible "no dset dictionary")
                   | (ref (DD mvar)) => SV.mGet mvar

      (* what dsets do we want to search? *)
      val dsets = findDSets (depth, dset, mydict, messenger)

      (* ok, from now on all we have to do is look at dsets *)
    in
      if (Dict.isEmpty dsets) then
	raise NoSuchDataset
      else 
	(Dict.appi searchOne dsets;
	 finish ())
    end

  (* should there be a subdataset? *)
  datatype what = LOCALSUB | REMOTE of string list | NOSUB

  (* recursively create from parents until it works *)
  (* val doCreate : dict * string -> dict * dataset *)
  fun doCreate ident (mydict, dset) =
    (let 
       fun setsubdataset (d, dsn, n) =
	 let
	   val sd = [{acl=NONE, attribute="subdataset", 
		      value = SOME (Value.List ["."])}]
	     
	   val (d, b) = doStore ident (mydict, dsn, d, NONE, n, sd)
	 in
	   if b = LOCALSUB then d
	   else raise (Impossible "it shouldn't have a subdataset?")
	 end

       fun doC (mydict, parent, name, nil) = (mydict, parent)
	 | doC (mydict, parent, name, _::nil) = (mydict, parent)
	 | doC (mydict, parent, name, a::l) =
	 let
	   val _ = if (a = "") 
		     then raise (SyntaxError "null dataset path member")
		   else ()
	   val myname = name ^ a ^ "/"
	   val _ = TextIO.print ("creating " ^ myname ^ "...\n")
	 in
	   case Dict.find (mydict, myname) of
	     NONE => (* this level doesn't exist *)
	       let
		 val _ = print ("loading parent: " ^ name ^ "\n")
		 val d' = SV.mTake parent
		 val d' = load (name, d')
		 val d' = setsubdataset (d', name, a)
		 val d' = doInheritance (name, mydict, d')
		 val _ = SV.mPut (parent, d')
		   
		 (* invariant: d' is now an InMem structure ! *)
		 val ({owner,inherits,acl,data,...}) = 
			      case d' of
				(InMem stuff) => stuff
			      | (Remote urls) => raise DatasetRefer urls
			      | (OnDisk _) => 
				  raise (Impossible "dataset not loaded")

		 val null = case (EL.find (data, "")) of
		   NONE => raise Impossible "dataset with null entry"
		 | (SOME e) => e
		     
		 val new = create (name, a, {null=null,
					     owner=ident,
					     inherits=inherits,
					     aclf=acl})

		 val new = SV.mVarInit (doInheritance (name, mydict, new))
	       in
		 doC (Dict.insert (mydict, myname, new), new, myname, l)
	       end
	   | (SOME d) => doC (mydict, d, myname, l)
	 end

      (* fetch the root dset *)
      val root = case Dict.find (mydict, "/") of
	NONE => raise (Impossible "no root dataset")
      | (SOME d) => d

      (* what's the path we have to create? *)
      val l = 
	case String.fields (fn c => c = #"/") dset of
	  nil => raise (SyntaxError "bad dataset path")
	| (a::l) => l
    in
      doC (mydict, root, "/", l)
    end handle exn => (print "Darn!\n"; raise exn))

  (* doStore returns the new dataset and whether the stored-to name
     should have a subdataset or not *)
  and doStore ident (mydict, dset, d, time, name as "", sd) =
    let 
      (* storing the null entry is much trickier, 
       since it can rock our world *)
      val _ = print "storing to null entry...\n"

      val d = load (dset, d)

      val {data, idata=oldidata,
	   channel,
	   dieListener,
	   inherits,
	   owner,
	   acl,
	   flag} = case d of
                     (Remote urls) => raise (DatasetRefer urls)
                   | (OnDisk _) => raise (DiskError "couldn't load dataset")
		   | (InMem stuff) => stuff

      val null = case EL.find (data, "") of
	NONE => raise (Impossible "no null entry")
      | (SOME e) => e

      val null = case (Entry.store ident acl null (sd, time)) of
	NONE => raise (SyntaxError "can't delete the null entry")
      | (SOME e) => e

      val data = EL.insert (data, "", null)
      val _ = writeSet (dset, data, addE("", null))

      (* find the new cache values *)
      val owner =
	case Entry.getattr' null "dataset.owner" of
	  NONE => raise (Entry.EnforcedValidation("", "dataset.owner"))
	| (SOME {value=V.Single owner,...}) => Auth.fromString owner
	| _ => raise (Entry.EnforcedValidation ("", "dataset.owner"))

      val inherits = 
	case Entry.getattr' null "dataset.inherit" of
	  NONE => NONE
	| (SOME {value=V.Single inherit,...}) => SOME inherit
	| (SOME {value=V.Nil,...}) => NONE
	| _ => raise (Entry.EnforcedValidation ("", "dataset.inherit"))

      fun acl attrname =
	case Entry.getattr' null ("dataset.acl." ^ attrname) of
	  NONE => (case Entry.getattr' null "dataset.acl" of
		     NONE => Acl.empty
		   | (SOME {value=aclval,...}) => Acl.fromValue aclval)
	| (SOME {value=aclval,...}) => Acl.fromValue aclval

      val d = InMem {data=data, idata=EL.empty,
		     channel=channel,
		     dieListener=NONE,
		     inherits=inherits,
		     owner=owner,
		     acl=acl,
		     flag=true}

      (* if we were listening to something, stop now *)
      val _ = 
	case dieListener of NONE => ()
                          | (SOME die) => SV.iPut (die, ())

      val d = doInheritance (dset, mydict, d)

      (* broadcast invalidating the old data *)
      val _ = EL.appi (fn (n, e) => MC.multicast (channel, 
						  Entry.DELETE(dset ^ n, e))) 
	                                         oldidata

      val {idata,...} = case d of
	(InMem stuff) => stuff
      | _ => raise (Impossible "huh?")

      (* broadcast the new data *)
      val _ = EL.appi (fn (n, e) => 
		       MC.multicast (channel, Entry.NEW(dset ^ n, e))) idata
    in
      (d, NOSUB) (* "" entries never have subdatasets *)
    end
    | doStore ident (mydict, dset, d, time, name, sd) =
    let (* storing a random entry is pretty straightforward *)
      val d = load (dset, d)
      val d = doInheritance (name, mydict, d)
	
      val fullname = dset ^ name

      val {data, idata,
	   channel,
	   dieListener,
	   inherits,
	   owner,
	   acl,
	   flag} = case d of
                     (Remote urls) => raise (DatasetRefer urls)
                   | (OnDisk _) => raise (Impossible "couldn't load dataset")
		   | (InMem stuff) => stuff
		     
      val oldentry = EL.find (idata, name)
      val oldentry' = EL.find (data, name)

      val newentry' =
	case oldentry' of
	  NONE => SOME (Entry.create ident acl name sd)
	| (SOME e) => Entry.store ident acl e (sd, time)

      val data = 
	case newentry' of
	  NONE => #1 (EL.remove (data, name))
	| (SOME e) => EL.insert (data, name, e)

      val _ = (* write it to disk *)
	case newentry' of
	  NONE => writeSet (dset, data, delE(name, AcapTime.now ()))
	| (SOME e) => writeSet (dset, data, addE(name, e))

      val subp = case newentry' of NONE => NOSUB
                                    | (SOME e) => 
	               case Entry.getattr' e "subdataset" of
			 (SOME {value=V.List ["."], ...}) => LOCALSUB
		       | (SOME {value=V.List l, ...}) => (REMOTE l)
		       | _ => NOSUB

      val d = InMem {data=data, idata=idata,
		     channel=channel,
		     dieListener=dieListener,
		     inherits=inherits,
		     owner=owner,
		     acl=acl,
		     flag=flag}

      val d = doInheritanceEntry (mydict, d, (name, newentry'))

      val {data, idata,
	   channel,
	   dieListener,
	   inherits,
	   owner,
	   acl,
	   flag} = case d of
                     (Remote urls) => raise (DatasetRefer urls)
                   | (OnDisk _) => raise (Impossible "couldn't load dataset")
		   | (InMem stuff) => stuff

      val newentry = EL.find (idata, name)

      val _ = case (oldentry, newentry) of
	(NONE, NONE) => ()
      | (NONE, SOME new) => MC.multicast (channel, Entry.NEW(fullname, new))
      | (SOME old, SOME new) => MC.multicast (channel, 
					      Entry.CHANGE((fullname, old),
							   (fullname, new)))
      | (SOME old, NONE) => MC.multicast (channel, Entry.DELETE(fullname, old))
    in
      (d, subp)
    end

  local
    fun create' ident (dsetname, parent as InMem{acl,inherits,data,...}) =
      let
	val ss = Substring.all dsetname
	val ss = Substring.dropr (fn c => c = #"/") ss
	val null = valOf (EL.find (data, ""))
	val (path, last) = Substring.splitr (fn c => c <> #"/") ss
	val new = create (Substring.string path, Substring.string last, 
			  {null=null, 
			   owner=ident, 
			   inherits=inherits,
			   aclf=acl})
      in
	new
      end

  in
    fun createIfNeeded (mydict, dsetname, ident, parent) = 
      case Dict.find (mydict, dsetname) of
	NONE => Dict.insert (mydict, dsetname, 
			     SV.mVarInit (create' ident (dsetname, parent)))
      | (SOME d) =>
	  let
	    val d' = SV.mTake d
	    val d' = case d' of
	      (Remote _) => create' ident (dsetname, parent)
	    | _ => d'
	  in
	    SV.mPut (d, d');
	    mydict
	  end

    fun deleteIfNeeded (mydict, dsetname) = 
      case Dict.find (mydict, dsetname) of
	NONE => mydict
      | (SOME d) => 
	  let
	    val d' = SV.mTake d
	    val mydict = delete (mydict, dsetname, d')
	  in
	    #1 (Dict.remove (mydict, dsetname))
	  end

    and delete (mydict, dsetname, d as Remote _) = mydict
      | delete (mydict, dsetname, d as OnDisk _) = 
      let
	val d = load (dsetname, d)
	val d = doInheritance (dsetname, mydict, d)
      in
        delete (mydict, dsetname, d)
      end 
      | delete (mydict, dsetname, d as InMem {idata, channel, ...}) =
      let
	fun announce (n, e, mydict) = 
	  (MC.multicast (channel, 
			 Entry.DELETE(dsetname ^ n, e));
	   deleteIfNeeded (mydict, dsetname ^ n ^ "/"))
			 
	val dir = Config.FDIR_spool ^ dsetname
	val file = Config.makeDataFile dsetname
	val mydict = EL.foldli announce mydict idata

      in
	Posix.FileSys.unlink file;
	Posix.FileSys.rmdir dir;
	mydict
      end

    fun remoteIfNeeded (mydict, dsetname, urls) =
      case Dict.find (mydict, dsetname) of
	NONE => Dict.insert (mydict, dsetname, SV.mVarInit (Remote urls))
      | (SOME d) => 
	  let
	    val d' = SV.mTake d
	      
	    val mydict = case d' of
	      (Remote _) => mydict
	    | _ => (delete (mydict, dsetname, d'))
	  in
	    SV.mPut (d, (Remote urls));
	    mydict
	  end
  end

  (*********
   * STORE *
   *********
   * val store : Entry.Acl.Auth.ident 
   *                 -> string * bool * AcapTime.acaptime option
   *                 -> (string * Entry.storedata list)
   *                 -> unit
   *
   * find the dataset:
   * - if docreate is set, create if necessary, otherwise abort
   *
   * attempt to store to the relevant entry
   *
   * decide if we're going to create a subdataset:
   * - "subdataset" is set to "."
   * decide if we're going to delete a subdataset:
   * - "subdataset" is set to NIL or DEFAULT
   * - OR "entry" is set to NIL or DEFAULT
   *)
  fun store ident (dset, docreate, time) (name, sd) = 
    let
      (* find the dataset:
       * - if docreate is set, create if necessary, otherwise abort *)
      val (mvar, olddict) = 
	case dict of (ref EMPTY) => raise (Impossible "no dset dictionary")
                   | (ref (DD mvar)) => (mvar, SV.mTake mvar)

      val (newdict, d) = 
	case Dict.find (olddict, dset) of
	  NONE => (print (dset ^ " doesn't exist...\n");
		   if docreate then 
		     doCreate ident (olddict, dset)
		   else raise NoSuchDataset)
	| (SOME d) => (olddict, d)

      (* ok, grab our dataset *)
      val d' = SV.mTake d
    in (* need containing context for exception handling *)
      let
	(* attempt to store to the relevant entry; may raise exception *)
	val (d', subp) = doStore ident (newdict, dset, d', time, name, sd)

	val _ = SV.mPut (d, d')

	(* the "doStore" would have raised an exception if there was a
	 conflict in the store.  

	 if subp then dset/name should exist
	 else it should be deleted *)
	val mysub = dset ^ name ^ "/"
	val newdict =
	  case subp of
	    NOSUB => deleteIfNeeded (newdict, mysub)
	  | LOCALSUB => createIfNeeded (newdict, mysub, ident, d')
	  | (REMOTE l) => remoteIfNeeded (newdict, mysub, l)
      in
	writeDSets newdict;
	SV.mPut (mvar, newdict)
      end handle exn => (SV.mPut (d, d');
			 SV.mPut (mvar, olddict);
			 raise exn)
    end

  (* acl stuff *)
  type aclobj = {dataset : string,
		 attribute : string option,
		 entry : string option}

  datatype aclmod = 
    SETACL of string * AclRights.rights
  | DELACL of string option
  | GETACL

  (* val modifyacl : Entry.Acl.Auth.ident -> (aclobj * aclmod) -> 
                     Entry.Acl.acl *)
  fun modifyacl ident ({dataset, attribute, entry}, aclmod) = 
    let 
      val mydict = 
	case dict of (ref EMPTY) => raise (Impossible "no dset dictionary")
                   | (ref (DD mvar)) => SV.mGet mvar

      val d = 
	case Dict.find (mydict, dataset) of 
	  NONE => raise NoSuchDataset
	| (SOME d) => d

      val d' = SV.mTake d
    in let
      val d' = load (dataset, d')
      val (data, aclf) = case d' of
	Remote urls => (SV.mPut (d, d');
			raise DatasetRefer urls)
      | (InMem {data,acl,...}) => (data, acl)
      | _ => (SV.mPut (d, d');
	      raise (Impossible "dataset not loaded"))

      val acl = 
	case attribute of
	  NONE => aclf ""
	| (SOME attribute) => 
	    case entry of
	      NONE => aclf attribute
	    | (SOME entry) => 
		(case EL.find (data, entry) of
		   NONE => raise (SyntaxError "no such entry")
		 | (SOME entry) => 
		     case (Option.mapPartial #acl 
			          (Entry.getattr' entry attribute)) of
		       NONE => aclf attribute
		     | (SOME acl) => acl)

      val acl = case aclmod of
	(SETACL update) => Acl.update acl update
      | (DELACL (SOME id)) => Acl.update acl (id, AclRights.empty)
      | (DELACL NONE) => Acl.empty
      | (GETACL) => acl

      val entryname = case entry of 
		   NONE => "" 
		 | (SOME e) => e

      val aclattr = case attribute of 
		   NONE => "dataset.acl"
		 | (SOME a) => "dataset.acl." ^ a

      val d' = case aclmod of
		   GETACL => d'
		 | _ => #1(doStore ident (mydict, dataset, d', NONE, entryname,
			     (case entry of
				NONE => [{acl=NONE, attribute=aclattr,
					  value=SOME (Acl.toValue acl)}]
			      | (SOME _) => [{acl=SOME acl, 
					      attribute=valOf attribute,
					      value=NONE}])))
       in SV.mPut (d, d'); acl
       end handle exn => (SV.mPut (d, d'); raise exn) end

end
