(* notify contexts should probably only queue events that are actually
 going to effect the context.

 also, it would be really nice if we don't send notifications that don't convey any information.  this is hard. *)
structure Context :> CONTEXT =
struct
  structure SV = LSyncVar
  structure MC = RMulticast

  structure Map = RedBlackMapFn(type ord_key = String.string
				val compare = String.compare)

  (* outgoing messages are batched up and sent no later than X seconds
   after generated *)
  val batchDelay = Time.fromSeconds 30

  datatype outMsg = 
    CNEW of string * Entry.entry * int
  | CCHANGE of string * Entry.entry * (int * int)
  | CDELETE of string * Entry.entry * int
  | CTIME of AcapTime.acaptime
  | CDIE

  exception ContextInconsistency
  exception Modified of AcapTime.acaptime
  exception NotNotifyContext

  datatype ctlMsg =
    E of Entry.updatemsg		(* entry to consider adding *)
  | U of unit SV.ivar			(* flush buffered updates,
					 issue modtime, and return *)
  | F					(* flush, noop if empty *)
  | Q					(* let the context die *)

  (* invariant: the enumeration (if it exists) holds whatever is currently 
                in the data map. *)
  type staticcontext = {data : Entry.entry Map.map,
			enumeration : (string * Entry.entry) Enum.enum option}
                              SV.mvar

  type notifycontext = {guard : Entry.entry -> bool,
			notify : outMsg CML.chan,
			listen : Entry.updatemsg MC.port list,
			lastupdate : AcapTime.acaptime,
			ctl : ctlMsg CML.chan,
			data : Entry.entry Map.map,
			enumeration : (string * Entry.entry) Enum.enum option}
                              SV.mvar

  datatype context = S of staticcontext | N of notifycontext

  fun defaultSort ((s1 : string, _), (s2 : string, _)) = String.compare(s1,s2)

  fun create _ (_, NONE, false) =
    S (SV.mVarInit {data=Map.empty, enumeration=NONE})
    | create _ (sort, NONE, true) =
    S (SV.mVarInit {data=Map.empty, 
		    enumeration=SOME (Enum.empty (getOpt(sort, defaultSort)))})
    | create guard (_, SOME notify, false) =
    N (SV.mVarInit {guard=guard, notify=notify, 
		    listen=nil, lastupdate=AcapTime.old,
		    ctl=CML.channel (),
		    data=Map.empty, enumeration=NONE})
    | create guard (sort, SOME notify, true) =
    N (SV.mVarInit {guard=guard, notify=notify, 
		    listen=nil, lastupdate=AcapTime.old,
		    ctl=CML.channel (),
		    data=Map.empty, 
		    enumeration=SOME (Enum.empty (getOpt(sort, defaultSort)))})

  (* if there's an enumeration, returns the new one and a position;
     otherwise NONE and zero *)
  fun oEnumIns (_, NONE) = (NONE, 0)
    | oEnumIns (a, SOME enum) = 
    let
      val (enum, pos) = Enum.insert (enum, a)
    in
      ((SOME enum), pos)
    end

  fun oEnumRem (_, NONE) = (NONE, 0)
    | oEnumRem (a, SOME enum) = 
    let
      val (enum, pos) = Enum.remove (enum, a)
    in
      ((SOME enum), pos)
    end

  fun fastInsert (S ctxt) new =
    let
      val {data, enumeration} = SV.mTake ctxt
    in
      SV.mPut (ctxt, {data=Map.insert' (new, data),
		      enumeration=(#1 (oEnumIns (new, enumeration)))})
    end
    | fastInsert (N ctxt) new =
    let
      val {guard, notify, listen, lastupdate, ctl, data, enumeration} = 
	SV.mTake ctxt
    in
      SV.mPut (ctxt, {guard=guard, notify=notify,
		      listen=listen, lastupdate=lastupdate,
		      ctl=ctl,
		      data=Map.insert' (new, data),
		      enumeration=(#1 (oEnumIns (new, enumeration)))})
    end

  (* attempt to insert new entry into context; give notification *)
  (* val insert : context -> string * Entry.entry -> unit *)
  fun insert (ctxt : notifycontext) (new as (name, newent)) =
    let
      val (c as {guard, notify, listen, lastupdate, ctl, data, enumeration}) = 
	SV.mTake ctxt
    in
      if guard newent then (* add it *)
	let
	  val data = Map.insert' (new, data)
	  val (enumeration, pos) = oEnumIns (new, enumeration)
	in
	  CML.send (notify, (CNEW(name, newent, pos)));
	  SV.mPut (ctxt, {guard=guard, notify=notify,
			  listen=listen, lastupdate=lastupdate,
			  ctl=ctl,
			  data=data, enumeration=enumeration})
	end
      else SV.mPut (ctxt, c)
    end

  (* change an entry that may or may not be in the context *)
  (* val change : context -> (string * Entry.entry) * (string * Entry.entry) 
    -> unit *)
  fun change (ctxt : notifycontext) (old as (oldname, oldent),
				     new as (newname, newent)) =
    let
      val (c as {guard, notify, listen, lastupdate, ctl, data, enumeration}) = 
	SV.mTake ctxt
    in
      if guard oldent then
	if guard newent then (* CHANGE *)
	  let
	    val (data, oldent) = (Map.remove (data, oldname)
		   handle LibBase.NotFound => raise ContextInconsistency)
	    val (enumeration, oldpos) = oEnumRem ((oldname, oldent), 
						  enumeration)
	    val data = Map.insert' (new, data)
	    val (enumeration, newpos) = oEnumIns (new, enumeration)
	  in
	    CML.send (notify, (CCHANGE(newname, newent, (oldpos, newpos))));
	    SV.mPut (ctxt, {guard=guard, notify=notify,
			    listen=listen, lastupdate=lastupdate,
			    ctl=ctl,
			    data=data, enumeration=enumeration})
	  end
	else (* DELETE *)
	  let
	    val (data, oldent) = (Map.remove (data, oldname)
			 handle LibBase.NotFound => raise ContextInconsistency)
	    val (enumeration, pos) = oEnumRem ((oldname, oldent), enumeration)
	  in
	    CML.send (notify, (CDELETE(oldname, oldent, pos)));
	    SV.mPut (ctxt, {guard=guard, notify=notify,
			    listen=listen, lastupdate=lastupdate,
			    ctl=ctl,
			    data=data, enumeration=enumeration})
	  end
      else 
	if guard newent then (* ADD *)
	  let
	    val data = Map.insert' (new, data)
	    val (enumeration, pos) = oEnumIns (new, enumeration)
	  in
	    CML.send (notify, (CNEW(newname, newent, pos)));
	    SV.mPut (ctxt, {guard=guard, notify=notify,
			    listen=listen, lastupdate=lastupdate,
			    ctl=ctl,
			    data=data, enumeration=enumeration})
	  end
	else (* NOOP *)
	  SV.mPut (ctxt, c)
    end

  (* delete an entry that may or may not be in the context *)
  (* val delete : context -> (string * Entry.entry) -> unit *)
  fun delete (ctxt : notifycontext) (old as (oldname, oldent)) =
    let
      val (c as {guard, notify, listen, lastupdate, ctl, data, enumeration}) = 
	SV.mTake ctxt
    in
      if guard oldent then
	let
	  val (data, oldent) = (Map.remove (data, oldname)
		       handle LibBase.NotFound => raise ContextInconsistency)
	  val (enumeration, pos) = oEnumRem ((oldname, oldent), enumeration)
	in
	  CML.send (notify, (CDELETE(oldname, oldent, pos)));
	  SV.mPut (ctxt, {guard=guard, notify=notify,
			  listen=listen, lastupdate=lastupdate,
			  ctl=ctl,
			  data=data, enumeration=enumeration})
	end
      else SV.mPut (ctxt, c)
    end

  fun processE ctx (Entry.NEW new) = insert ctx new
    | processE ctx (Entry.CHANGE chg) = change ctx chg
    | processE ctx (Entry.DELETE old) = delete ctx old

  fun sendTime (ctxt : notifycontext) =
    let
      val ({notify,lastupdate,...}) = SV.mGet ctxt
    in
      CML.send (notify, CTIME(lastupdate))
    end

  fun updateTime (ctxt : notifycontext) =
    let
      val (c as {guard, notify, listen, lastupdate, ctl, data, enumeration}) = 
	SV.mTake ctxt

      val lastupdate = AcapTime.now ()
    in
      CML.send (notify, CTIME(lastupdate));
      SV.mPut (ctxt, {guard=guard, notify=notify,
		      listen=listen, lastupdate=lastupdate,
		      ctl=ctl,
		      data=data, enumeration=enumeration})
    end

  fun listener (ctx, evts, ctl) = 
    let
      (* process all the entry events waiting *)
      val process = Fifo.app (processE ctx)

      fun timerThread () = (CML.sync(CML.timeOutEvt batchDelay);
			    CML.send (ctl, F))
	
      (* two-state listener; we queue up events until we get a flush
       or we time out *)
      fun empty () =
	case CML.select evts of
	  (E x) => (CML.spawn timerThread; 
		    nonempty (Fifo.enqueue (Fifo.empty, x)))
	| (U sv) => (sendTime ctx; 
		     SV.iPut (sv, ()); 
		     empty ())
	| F => (* already flushed! *) (empty ())
	| Q => ()
      and nonempty fifo =
	case CML.select evts of
	  (E x) => nonempty (Fifo.enqueue (fifo, x))
	| (U sv) => (process fifo; 
		     updateTime ctx; 
		     SV.iPut (sv, ()); 
		     empty ())
	| F => (process fifo; 
		updateTime ctx; 
		empty ())
	| Q => ()
    in
      empty ()
    end

  fun startListener (ctxt, ctl, listen) =
    let
      val ctlEvt = CML.recvEvt ctl
      val evts = List.map (fn p => CML.wrap (MC.recvEvt p, E)) listen
    in
      ignore (CML.spawnc listener (ctxt, ctlEvt::evts, ctl))
    end

  fun doneInitial (S ctxt) = ()
    | doneInitial (N ctxt) =
    let
      val (c as {guard, notify, listen, lastupdate, ctl, data, enumeration}) = 
	SV.mTake ctxt

      val _ = startListener (ctxt, ctl, listen)
    in
      SV.mPut (ctxt, {guard=guard, notify=notify,
		      listen=listen, lastupdate=lastupdate,
		      ctl=ctl,
		      data=data, enumeration=enumeration})
    end

  fun updatecontext (N ctxt) = 
    let
      val ({ctl,...}) = SV.mGet ctxt
      val sv = SV.iVar ()
    in 
      CML.send (ctl, U sv);
      SV.iGet sv
    end
    | updatecontext (S _) = ()

  (* add a new dataset to listen to *)
  (* val addDSet : context -> Entry.updatemsg RMulticast.port -> unit *)
  fun addDSet (N ctxt) port =
    let
      val (c as {guard, notify, listen, lastupdate, ctl, data, enumeration}) = 
	SV.mTake ctxt

      val listen = port::listen
    in
      SV.mPut (ctxt, {guard=guard, notify=notify,
		      listen=listen, lastupdate=lastupdate,
		      ctl=ctl,
		      data=data, enumeration=enumeration})
    end
    | addDSet (S _) _ = ()

  fun destroy (N ctxt) =
    let
      val (c as {guard, notify, listen, lastupdate, ctl, data, enumeration}) = 
	SV.mTake ctxt
    in
      CML.send (ctl, Q);
      CML.send (notify, CDIE);
      List.app MC.release listen
    end
    | destroy (S ctxt) = ()

  (* val search : context -> ((int * Entry.entry) -> bool) * 
      (string * Entry.entry) Throttle.throttle * AcapTime.acaptime -> unit *)
  local 
    fun searchData (data, lastupdate) (sfun, out, time) =
      let
	val out' = Throttle.call out
	fun win (thing as (name, entry)) = 
	  if sfun (0, entry) then out' thing else ()
      in
      if AcapTime.compare (time, lastupdate) = LESS then
	raise (Modified lastupdate)
      else (Map.appi win data;
	    Throttle.done out)
      end
    
    fun searchEnum (enum, lastupdate) (sfun, out, time) =
      let
	val out' = Throttle.call out
	fun win (thing as (name, entry), pos) =
	  if sfun (pos, entry) then out' thing else ()
      in
	if AcapTime.compare (time, lastupdate) = LESS then
	  raise (Modified lastupdate)
	else (Enum.app win enum;
	      Throttle.done out)
      end
  in
    fun search (N ctxt) sparams =
      let
	val ({lastupdate, data, enumeration, ...}) = SV.mGet ctxt
      in
	case enumeration of
	  NONE => searchData (data, lastupdate) sparams
	| (SOME enum) => searchEnum (enum, lastupdate) sparams
      end
      | search (S ctxt) sparams = 
      let
	val ({data, enumeration, ...}) = SV.mGet ctxt
      in
	case enumeration of
	  NONE => searchData (data, AcapTime.old) sparams
	| (SOME enum) => searchEnum (enum, AcapTime.old) sparams
      end
  end

end
