(* $Id: aclrights.sml,v 1.1 2000/02/25 02:44:34 leg Exp $ *)

signature ACLRIGHTS =
sig
  type rights

  exception InvalidRights

  val ACLx : rights
  val ACLr : rights
  val ACLw : rights
  val ACLi : rights
  val ACLa : rights
  val empty : rights

  val has : rights * rights -> bool

  val union : rights * rights -> rights

  val diff : rights * rights -> rights

  val toString : rights -> string

  val fromString : string -> rights
end

structure AclRights :> ACLRIGHTS =
struct
  structure W = Word32

  type rights = W.word

  exception InvalidRights

  val ACLx = 0wx01 : W.word
  val ACLr = 0wx02 : W.word
  val ACLw = 0wx04 : W.word
  val ACLi = 0wx08 : W.word
  val ACLa = 0wx10 : W.word

  val empty = 0wx0 : W.word

  infix has

  fun (rights has flag) = W.andb(rights, flag) <> 0wx0

  fun union (rights, flag) = W.orb(rights, flag)

  fun diff (rights, flag) = W.andb(rights, W.notb flag)

  fun toString rights = 
    (if rights has ACLx then "x" else "")
  ^ (if rights has ACLr then "r" else "")
  ^ (if rights has ACLw then "w" else "")
  ^ (if rights has ACLi then "i" else "")
  ^ (if rights has ACLa then "a" else "")

  fun fromString' (nil, acc) = acc
    | fromString' (#"x"::l, acc) = fromString' (l, union (acc, ACLx))
    | fromString' (#"r"::l, acc) = fromString' (l, union (acc, ACLr))
    | fromString' (#"w"::l, acc) = fromString' (l, union (acc, ACLw))
    | fromString' (#"i"::l, acc) = fromString' (l, union (acc, ACLi))
    | fromString' (#"a"::l, acc) = fromString' (l, union (acc, ACLa))
    | fromString' _ = raise InvalidRights

  fun fromString rights = fromString' (String.explode rights, empty)

end
