/* skip-list.c -- generic skip list routines
 *
 * Copyright (c) 1998, 2000 Carnegie Mellon University.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdlib.h>
#include <values.h>
#include "skip-list.h"

struct Skiplist {
    int maxlevel;
    float prob;
    int curlevel;
    int items;
    int (*comp)(const void *, const void *);
    skipnode *header; /* phantom first element of skiplist */
#if 0
    skipnode *finger[SKIP_ABSMAXLVL]; /* finger for localization */
#endif
};

skiplist *skiplist_new(int ml, float p, int (*cf)(const void *, const void *))
{
    skiplist *s = (skiplist *) malloc(sizeof(skiplist));
    int i;

    if (ml > SKIP_ABSMAXLVL) 
	ml = SKIP_ABSMAXLVL;
    s->maxlevel = ml;
    s->prob = p;
    s->comp = cf;
    s->items = 0;

    s->header = (skipnode *) malloc(sizeof(skipnode) + 
				    ml * sizeof(skipnode *));
    s->header->data = NULL;
    for (i = 0; i < ml; i++)
	s->header->forward[i] = NULL;

    s->curlevel = 0;

    return s;
}

void *ssearch(skiplist *S, void *k)
{
    skipnode *x = S->header;
    int i;
    int lvl;

    lvl = 1;

    /* invariant: x < k */
    for (i = lvl; i >= 0; i--) {
	while ((x->forward[i] != NULL) && 
	       (S->comp(x->forward[i]->data, k) < 0)) {
	    x = x->forward[i];
	}
    }
    x = x->forward[0];
    if ((x != NULL) && (S->comp(x->data, k) == 0))
	return x->data;
    else
	return NULL;
}

int randLevel(skiplist *S)
{
    int lvl = 0;
    while ((((float) rand() / (float) (MAXINT)) < S->prob) 
	   && (lvl < S->maxlevel))
	lvl++;
    return lvl;
}

void sinsert(skiplist *S, void *k)
{
    skipnode *update[SKIP_ABSMAXLVL];
    int newlvl = randLevel(S), i;
    skipnode *x = S->header, 
	*new = (skipnode *) malloc(sizeof(skipnode) +
				   newlvl * sizeof(skipnode *));
  
    for (i = S->curlevel; i >= 0; i--) {
	while (x->forward[i] && S->comp(x->forward[i]->data, k) < 0)
	    x = x->forward[i];
	update[i] = x;
    }
    x = x->forward[0];
    if ((x != NULL) && (S->comp(x->data, k) == 0)) {
	x->data = k;
	free(new);
    } else {
	S->items++;
	if (newlvl > S->curlevel)
	    for (i = S->curlevel + 1; i <= newlvl; i++)
		update[i] = S->header;
	new->data = k;
	for (i = 0; i <= newlvl; i++) {
	    new->forward[i] = update[i]->forward[i];
	    update[i]->forward[i] = new;
	}
    }
}

void sdelete(skiplist *S, void *k)
{
    skipnode *update[SKIP_ABSMAXLVL];
    skipnode *x = S->header;
    int i;
 
    for (i = S->curlevel; i >= 0; i--) {
	while (x->forward[i] && S->comp(x->forward[i]->data, k) < 0)
	    x = x->forward[i];
	update[i] = x;
    }
    x = x->forward[0];
    if ((x != NULL) && (S->comp(x->data, k) == 0)) {
	for (i = 0; i <= S->curlevel; i++) {
	    if (update[i]->forward[i] != x) break;
	    update[i]->forward[i] = x->forward[i];
	}
	free(x);
	S->items--;
	while ((S->curlevel >= 1) && (S->header->forward[S->curlevel] == NULL))
	    S->curlevel--;
    }

    /* if it isn't here, do nothing */
}

void skiplist_free(skiplist *S)
{
    skipnode *x = S->header, *y;

    while (x != NULL) {
	y = x->forward[0];
	free(x);
	x = y;
    }
    free(S);
}

int sempty(skiplist *S)
{
    return (S->items == 0);
}

int skiplist_items(skiplist *S)
{
    return S->items;
}

void sforeach(skiplist *S, void (*f)(const void *))
{
    skipnode *s;

    s = S->header->forward[0];
    while (s != NULL) {
	f(s->data);
	s = s->forward[0];
    }
}

void *sfirst(skiplist *S, skipnode **ptr)
{
    *ptr = S->header->forward[0];
    if (*ptr != NULL) {
	return (*ptr)->data;
    } else {
	return NULL;
    }
}

void *snext(skipnode **ptr)
{
    if (*ptr != NULL) {
	*ptr = (*ptr)->forward[0];
    }
    if (*ptr != NULL) {
	return (*ptr)->data;
    } else {
	return NULL;
    }
}
